"""
Runs and tests everything in the examples folder.
"""

import pytest
import glob
import runpy
import os
import psyneulink
import sys
import copy


from distutils.dir_util import copy_tree
from pathlib import Path

example_scripts = glob.glob("examples/**/*.py", recursive=True)
example_mdf_scripts = {
    Path(f) for f in glob.glob("examples/MDF/**/*.py", recursive=True)
}
example_pnl_scripts = {
    Path(f) for f in glob.glob("examples/PsyNeuLink/**/*.py", recursive=True)
}
example_exclusion_strings = [".reconstructed.py", "generate_json_and_scripts.py"]

# NOTE: xfail mark can be removed if/when a full TimeInterval or
# placeholder class reaches main psyneulink branch. This is only a
# concern for local/non-github-actions installations.
example_scripts = [
    pytest.param(
        script,
        marks=pytest.mark.xfail(
            reason="psyneulink currently must be installed on its 'mdf' branch to work with MDF examples",
            strict=False,
        ),
    )
    if (Path(script) in example_pnl_scripts and not hasattr(psyneulink, "TimeInterval"))
    else script
    for script in example_scripts
    if all(e not in script for e in example_exclusion_strings)
]


@pytest.fixture(autouse=True)
def chdir_back_to_root(mocker):
    """
    This fixture sets up and tears down state before each example is run. Certain examples
    require that they are run from the local directory in which they reside. This changes
    directory and adds the local directory to sys.path. It reverses this after the test
    finishes.
    """

    # Get the current directory before running the test
    cwd = os.getcwd()

    # Some of the scripts do plots. Lets patch matplotlib plot so tests don't hang
    mocker.patch("matplotlib.pyplot.show")
    mocker.patch("matplotlib.pyplot.figure")

    # Cache the path so we can reset it after the test, the examples/MDF tests require
    # setting the paht
    old_sys_path = copy.copy(sys.path)

    yield

    sys.path = old_sys_path

    # We need chdir back to root of the repo
    os.chdir(cwd)


@pytest.mark.parametrize("script", example_scripts)
@pytest.mark.parametrize("additional_args", [["-run"]])
def test_example(script, tmpdir, additional_args):
    """
    Run the examples/MDF
    """

    # Get the full path for the script
    full_script_path = os.path.abspath(script)

    # Some of the scripts in examples/MDF import from the local directory. So lets run from the scripts
    # local directory.
    dir_path = os.path.dirname(os.path.realpath(full_script_path))

    # If this is one of the example/MDF scripts, we need to append example/MDF to sys.path
    if Path(script) in example_mdf_scripts:
        sys.path.append(dir_path)

    # Copy the contents of this directory to a tmpdir (so any files generated by the example will get cleaned up when
    # we run it)
    if "translation" not in dir_path:
        copy_tree(dir_path, tmpdir.strpath)
        os.chdir(tmpdir)
    else:
        copy_tree("%s/.." % dir_path, tmpdir.strpath)
        os.chdir("%s/translation" % tmpdir)

    print(f"Running script {full_script_path} in working dir {os.getcwd()}")
    orig_argv = sys.argv
    sys.argv = [os.path.basename(full_script_path)] + additional_args
    runpy.run_path(os.path.basename(full_script_path), run_name="__main__")
    sys.argv = orig_argv
