"""Initialization functions to create a GE individual."""

from .._shared import init_individual as _init_individual
from . import default_parameters as _dp
from . import mapping as _mapping
from . import representation as _representation


def given_genotype(grammar, parameters=None):
    """Create an individual from a given genotype.

    Parameters
    ----------
    grammar : `~alogos.Grammar`
    parameters : `dict` or `~alogos._utilities.parametrization.ParameterCollection`, optional
        Following keyword-value pairs are considered by this function:

        - ``init_ind_given_genotype`` : Data for a GE
          `~.representation.Genotype`.

    Raises
    ------
    InitializationError
        If the initialization of the individual fails.

    Notes
    -----
    The genotype is converted to a derivation tree and phenotype with
    the `~.mapping.forward` mapping function of this system.

    """
    return _init_individual.given_genotype(
        grammar, parameters, _dp, _representation, _mapping
    )


def given_derivation_tree(grammar, parameters=None):
    """Create an individual from a given derivation tree.

    Parameters
    ----------
    grammar : `~alogos.Grammar`
    parameters : `dict` or `~alogos._utilities.parametrization.ParameterCollection`, optional
        Following keyword-value pairs are considered by this function:

        - ``init_ind_given_derivation_tree`` : Data for a
          `~alogos._grammar.data_structures.DerivationTree`.

    Raises
    ------
    InitializationError
        If the initialization of the individual fails.

    Notes
    -----
    The leaf nodes of the derivation tree are read to create the
    phenotype.
    The phenotype is converted to a genotype with
    the `~.mapping.reverse` mapping function of this system.

    """
    return _init_individual.given_derivation_tree(
        grammar, parameters, _dp, _representation, _mapping
    )


def given_phenotype(grammar, parameters=None):
    """Create an individual from a given phenotype.

    Parameters
    ----------
    grammar : `~alogos.Grammar`
    parameters : `dict` or `~alogos._utilities.parametrization.ParameterCollection`, optional
        Following keyword-value pairs are considered by this function:

        - ``init_ind_given_phenotype`` (`str`) : Data for a phenotype,
          which needs to be a string of the grammar's language.

    Raises
    ------
    InitializationError
        If the initialization of the individual fails.

    Notes
    -----
    The phenotype is converted to a genotype and derivation tree with
    the `~.mapping.reverse` mapping function of this system.

    """
    return _init_individual.given_phenotype(
        grammar, parameters, _dp, _representation, _mapping
    )


def random_genotype(grammar, parameters=None):
    """Create an individual from a random genotype.

    Parameters
    ----------
    grammar : `~alogos.Grammar`
    parameters : `dict` or `~alogos._utilities.parametrization.ParameterCollection`, optional
        Following keyword-value pairs are considered by this function:

        - ``genotype_length`` (`int`) : Number of integers in the GE
          `~.representation.Genotype`.
        - ``codon_size`` (`int`) : Number of bits used for a codon,
          which determines the maximum integer value a codon can take.
          For example, a codon size of 8 bits allows integers from
          0 to 255 (from 2**8-1).

    Raises
    ------
    InitializationError
        If the initialization of the individual fails.

    Notes
    -----
    The random genotype is a fixed-length `list` of `int` which are
    drawn independently from a uniform distribution of numbers in the
    interval ``[0, 2**codon_size)``.

    References
    ----------
    - 2016, Nicolau, Fenton: `Managing Repetition in Grammar-Based Genetic Programming
      <https://doi.org/10.1145/2908812.2908904>`__

    - 2017, Nicolau: `Understanding grammatical evolution: initialisation
      <https://doi.org/10.1007/s10710-017-9309-9>`__

        - 3.1 Random initialisation (RND)

    """
    return _init_individual.random_genotype(
        grammar, parameters, _dp, _representation, _mapping
    )


def random_valid_genotype(grammar, parameters=None):
    """Create an individual from a random genotype likely to be valid.

    Parameters
    ----------
    grammar : `~alogos.Grammar`
    parameters : `dict` or `~alogos._utilities.parametrization.ParameterCollection`, optional
        Following keyword-value pairs are considered by this function:

        - ``genotype_length`` (`int`) : Number of integers in the GE
          `~.representation.Genotype`.
        - ``codon_size`` (`int`) : Number of bits used for a codon,
          which determines the maximum integer value a codon can take.
          For example, a codon size of 8 bits allows integers from
          0 to 255 (from 2**8-1).
        - ``init_ind_random_valid_genotype_max_tries`` (`int`) : Number
          of tries to generate a random valid genotype.

    Raises
    ------
    InitializationError
        If the initialization of the individual fails.

    Notes
    -----
    This function repeatedly calls `random_genotype` until it returns
    a genotype that can be mapped to a phenotype or a given maximum
    number of tries is reached.

    """
    return _init_individual.random_valid_genotype(
        grammar, parameters, _dp, _representation, _mapping
    )


def gp_grow_tree(grammar, parameters=None):
    """Create an individual from a randomly grown tree."""
    return _init_individual.gp_grow_tree(
        grammar, parameters, _dp, _representation, _mapping
    )


def pi_grow_tree(grammar, parameters=None):
    """Create an individual from a position-independently, randomly grown tree."""
    return _init_individual.pi_grow_tree(
        grammar, parameters, _dp, _representation, _mapping
    )


def gp_full_tree(grammar, parameters=None):
    """Create an individual from a fully grown tree up to a maximum depth."""
    return _init_individual.gp_full_tree(
        grammar, parameters, _dp, _representation, _mapping
    )


def ptc2_tree(grammar, parameters=None):
    """Create an individual from a tree grown with Nicolau's "PTC2".

    The original PTC2 method for growing random trees was invented by
    Sean Luke in 2000. Some slightly modified variants were
    introduced later by other authors.
    This function implements a PTC2 variant described by Miguel Nicolau
    in 2017 in section "3.3 Probabilistic tree-creation 2 (PTC2)" of
    the publication. It restricts tree size not with a maximum tree
    depth but rather with a maximum number of expansions and if
    possible remains strictly below this limit.

    Parameters
    ----------
    grammar : `~alogos.Grammar`
    parameters : `dict` or `~alogos._utilities.parametrization.ParameterCollection`, optional
        Following keyword-value pairs are considered by this function:

        - ``init_ind_ptc2_max_expansions`` (`int`): The maximum number
          of nonterminal expansions that may be used to grow the tree.

    Raises
    ------
    InitializationError
        If the initialization of the individual fails.

    Notes
    -----
    The leaf nodes of the derivation tree are read to create the
    phenotype.
    The phenotype is converted to a genotype with
    the `~.mapping.reverse` mapping function of this system.

    References
    ----------
    - 2000, Luke: `Two Fast Tree-Creation Algorithms for Genetic
      Programming <https://doi.org/10.1109/4235.873237>`__

        - "PTC1 is a modification of GROW that allows the user to
          provide probabilities of appearance of functions in the tree,
          plus a desired expected tree size, and guarantees that, on
          average, trees will be of that size."

        - "With PTC2, the user provides a probability distribution of
          requested tree sizes. PTC2 guarantees that, once it has picked
          a random tree size from this distribution, it will generate
          and return a tree of that size or slightly larger."

    - 2010, Harper: `GE, explosive grammars and the lasting legacy of
      bad initialisation
      <https://doi.org/10.1109/CEC.2010.5586336>`__

        - "The PTC2 methodology is extended to GE and found to produce
          a more uniform distribution of parse trees."

        - "If the algorithm is called in a ramped way (i.e. starting
          with a low number of expansions, say 20, and increasing until
          say 240) then a large number of trees of different size and
          shapes will be generated."

    - 2017, Nicolau: `Understanding grammatical evolution:
      initialisation <https://doi.org/10.1007/s10710-017-9309-9>`__:

        - 3.3 Probabilistic tree-creation 2 (PTC2)
        - 3.6 Probabilistic tree-creation 2 with depth limit (PTC2D)

    - 2018, Ryan, O'Neill, Collins: `Handbook of Grammatical Evolution
      <https://doi.org/10.1007/978-3-319-78717-6>`__

        - p. 13: "More recent work on initialisation includes that of
          Nicolau, who demonstrated that across the problems examined
          in their study, a variant of Harper’s PTC2 consistently
          outperforms other initialisations"

    """
    return _init_individual.ptc2_tree(
        grammar, parameters, _dp, _representation, _mapping
    )
