"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateCloudfrontSite = exports.CreateBasicSite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const acm = require("@aws-cdk/aws-certificatemanager");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const route53 = require("@aws-cdk/aws-route53");
const targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
/**
 * @stability stable
 */
class CreateBasicSite extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const hostedZoneLookup = route53.HostedZone.fromLookup(this, 'WebsiteHostedZone', {
            domainName: props.hostedZone,
        });
        const websiteRedirectBucket = new s3.Bucket(scope, 'WebsiteRedirectBucket', {
            bucketName: `www.${props.hostedZone}`,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            websiteRedirect: {
                hostName: props.hostedZone,
                protocol: s3.RedirectProtocol.HTTP,
            },
        });
        const websiteBucket = new s3.Bucket(scope, 'WebsiteBucket', {
            bucketName: props.hostedZone,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            websiteIndexDocument: props.indexDoc,
            websiteErrorDocument: props.errorDoc,
            publicReadAccess: true,
            encryption: s3.BucketEncryption.S3_MANAGED,
        });
        new s3deploy.BucketDeployment(scope, 'WebsiteDeploy', {
            sources: [s3deploy.Source.asset(props.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        new route53.ARecord(scope, 'WebisteAlias', {
            zone: hostedZoneLookup,
            recordName: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(websiteBucket)),
        });
        new route53.ARecord(scope, 'WebisteRedirectAlias', {
            zone: hostedZoneLookup,
            recordName: `www.${props.hostedZone}`,
            target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(websiteRedirectBucket)),
        });
    }
}
exports.CreateBasicSite = CreateBasicSite;
_a = JSII_RTTI_SYMBOL_1;
CreateBasicSite[_a] = { fqn: "cdk-simplewebsite-deploy.CreateBasicSite", version: "0.4.57" };
/**
 * @stability stable
 */
class CreateCloudfrontSite extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.domain && props.subDomain) {
            throw new Error('Domain and sub domain parameters cannot both be defined');
        }
        const hostedZoneLookup = route53.HostedZone.fromLookup(this, 'WebsiteHostedZone', {
            domainName: props.hostedZone,
        });
        const errorResponses = [];
        if (props.errorDoc) {
            errorResponses.push({
                httpStatus: 404,
                responsePagePath: `/${props.errorDoc}`,
            });
            errorResponses.push({
                httpStatus: 403,
                responsePagePath: `/${props.errorDoc}`,
            });
        }
        else {
            errorResponses.push({
                httpStatus: 404,
                responseHttpStatus: 200,
                responsePagePath: `/${props.indexDoc}`,
            });
            errorResponses.push({
                httpStatus: 403,
                responseHttpStatus: 200,
                responsePagePath: `/${props.indexDoc}`,
            });
        }
        const subjectAlternativeNames = [];
        if (props.domain)
            subjectAlternativeNames.push(props.domain);
        if (props.subDomain)
            subjectAlternativeNames.push(props.subDomain);
        const websiteCert = new acm.DnsValidatedCertificate(this, 'WebsiteCert', {
            domainName: props.hostedZone,
            subjectAlternativeNames,
            hostedZone: hostedZoneLookup,
            region: 'us-east-1',
        });
        const websiteBucket = new s3.Bucket(scope, 'WebsiteBucket', {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            publicReadAccess: false,
            encryption: s3.BucketEncryption.S3_MANAGED,
        });
        const domainNames = [];
        if (props.domain) {
            domainNames.push(props.domain);
        }
        else {
            domainNames.push(props.hostedZone);
        }
        if (props.subDomain)
            domainNames.push(props.subDomain);
        const websiteDist = new cloudfront.Distribution(scope, 'WebsiteDist', {
            defaultBehavior: {
                origin: new origins.S3Origin(websiteBucket),
                allowedMethods: cloudfront.AllowedMethods.ALLOW_GET_HEAD,
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2019,
            priceClass: props.priceClass
                ? props.priceClass
                : cloudfront.PriceClass.PRICE_CLASS_100,
            errorResponses,
            defaultRootObject: props.indexDoc,
            domainNames,
            certificate: websiteCert,
        });
        new s3deploy.BucketDeployment(scope, 'WebsiteDeploy', {
            sources: [s3deploy.Source.asset(props.websiteFolder)],
            destinationBucket: websiteBucket,
            distribution: websiteDist,
        });
        new route53.ARecord(scope, 'WebisteDomainAlias', {
            zone: hostedZoneLookup,
            recordName: props.domain ? props.domain : props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(websiteDist)),
        });
        if (props.subDomain) {
            new route53.ARecord(scope, 'WebisteSubDomainAlias', {
                zone: hostedZoneLookup,
                recordName: props.subDomain,
                target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(websiteDist)),
            });
        }
    }
}
exports.CreateCloudfrontSite = CreateCloudfrontSite;
_b = JSII_RTTI_SYMBOL_1;
CreateCloudfrontSite[_b] = { fqn: "cdk-simplewebsite-deploy.CreateCloudfrontSite", version: "0.4.57" };
//# sourceMappingURL=data:application/json;base64,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