import { Construct } from 'constructs';
import { ResourceProps, Resource } from './base';
import * as cdk8s from 'cdk8s';
/**
 * Properties for initialization of `Service`.
 */
export interface ServiceProps extends ResourceProps {
    /**
     * The spec of the service. Use `service.spec` to apply post instantiation mutations.
     *
     * @default - An empty spec will be created.
     */
    readonly spec?: ServiceSpec;
}
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
export declare enum ServiceType {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    CLUSTER_IP = "ClusterIP",
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    NODE_PORT = "NodePort",
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    LOAD_BALANCER = "LoadBalancer",
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    EXTERNAL_NAME = "ExternalName"
}
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
export declare class Service extends Resource {
    protected readonly apiObject: cdk8s.ApiObject;
    /**
     * Provides access to the underlying spec.
     *
     * You can use this field to apply post instantiation mutations
     * to the spec.
     */
    readonly spec: ServiceSpecDefinition;
    constructor(scope: Construct, id: string, props?: ServiceProps);
}
export declare enum Protocol {
    TCP = "TCP",
    UDP = "UDP",
    SCTP = "SCTP"
}
export interface ServicePortOptions {
    /**
     * The name of this port within the service. This must be a DNS_LABEL. All
     * ports within a ServiceSpec must have unique names. This maps to the 'Name'
     * field in EndpointPort objects. Optional if only one ServicePort is defined
     * on this service.
     */
    readonly name?: string;
    /**
     * The port on each node on which this service is exposed when type=NodePort
     * or LoadBalancer. Usually assigned by the system. If specified, it will be
     * allocated to the service if unused or else creation of the service will
     * fail. Default is to auto-allocate a port if the ServiceType of this Service
     * requires one.
     *
     * @see https://kubernetes.io/docs/concepts/services-networking/service/#type-nodeport
     *
     * @default to auto-allocate a port if the ServiceType of this Service
     * requires one.
     */
    readonly nodePort?: number;
    /**
     * The IP protocol for this port. Supports "TCP", "UDP", and "SCTP". Default is TCP.
     *
     * @default Protocol.TCP
     */
    readonly protocol?: Protocol;
    /**
     * The port number the service will redirect to.
     *
     * @default - The value of `port` will be used.
     */
    readonly targetPort?: number;
}
/**
 * Definition of a service port.
 */
export interface ServicePort extends ServicePortOptions {
    /**
     * The port number the service will bind to.
     */
    readonly port: number;
}
/**
 * Properties for initialization of `ServiceSpec`.
 */
export interface ServiceSpec {
    /**
     * The IP address of the service and is usually assigned randomly by the
     * master. If an address is specified manually and is not in use by others, it
     * will be allocated to the service; otherwise, creation of the service will
     * fail. This field can not be changed through updates. Valid values are
     * "None", empty string (""), or a valid IP address. "None" can be specified
     * for headless services when proxying is not required. Only applies to types
     * ClusterIP, NodePort, and LoadBalancer. Ignored if type is ExternalName.
     *
     * @see https://kubernetes.io/docs/concepts/services-networking/service/#virtual-ips-and-service-proxies
     * @default - Automatically assigned.
     *
     */
    readonly clusterIP?: string;
    /**
     * A list of IP addresses for which nodes in the cluster will also accept
     * traffic for this service. These IPs are not managed by Kubernetes. The user
     * is responsible for ensuring that traffic arrives at a node with this IP. A
     * common example is external load-balancers that are not part of the
     * Kubernetes system.
     *
     * @default - No external IPs.
     */
    readonly externalIPs?: string[];
    /**
     * Determines how the Service is exposed.
     *
     * More info: https://kubernetes.io/docs/concepts/services-networking/service/#publishing-services-service-types
     *
     * @default ServiceType.ClusterIP
     */
    readonly type?: ServiceType;
    /**
     * The port exposed by this service.
     *
     * More info: https://kubernetes.io/docs/concepts/services-networking/service/#virtual-ips-and-service-proxies
     */
    readonly ports?: ServicePort[];
}
/**
 * A description of a service.
 */
export declare class ServiceSpecDefinition {
    /**
     * The IP address of the service and is usually assigned randomly by the
     * master.
     */
    readonly clusterIP?: string;
    /**
     * A list of IP addresses for which nodes in the cluster will also accept
     * traffic for this service.
     */
    private readonly externalIPs;
    /**
     * Determines how the Service is exposed.
     */
    readonly type: ServiceType;
    private readonly _selector;
    private readonly _ports;
    constructor(props?: ServiceSpec);
    /**
     * Returns the labels which are used to select pods for this service.
     */
    get selector(): Record<string, string>;
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     */
    addSelector(label: string, value: string): void;
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    serve(port: number, options?: ServicePortOptions): void;
}
