"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAccount = void 0;
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const cdk8s = require("cdk8s");
const utils_1 = require("./utils");
/**
 * A service account provides an identity for processes that run in a Pod.
 *
 * When you (a human) access the cluster (for example, using kubectl), you are
 * authenticated by the apiserver as a particular User Account (currently this
 * is usually admin, unless your cluster administrator has customized your
 * cluster). Processes in containers inside pods can also contact the apiserver.
 * When they do, they are authenticated as a particular Service Account (for
 * example, default).
 *
 * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account
 */
class ServiceAccount extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, props);
        this._secrets = (_a = props.secrets) !== null && _a !== void 0 ? _a : [];
        this.apiObject = new k8s.ServiceAccount(this, 'Resource', {
            metadata: props.metadata,
            secrets: cdk8s.Lazy.any({ produce: () => utils_1.undefinedIfEmpty(this._secrets.map(s => ({ name: s.name }))) }),
        });
    }
    /**
     * Imports a service account from the cluster as a reference.
     * @param name The name of the service account resource.
     */
    static fromServiceAccountName(name) {
        return { name: name };
    }
    /**
     * Allow a secret to be accessed by pods using this service account.
     * @param secret The secret
     */
    addSecret(secret) {
        this._secrets.push(secret);
    }
    /**
     * List of secrets allowed to be used by pods running using this service
     * account.
     *
     * Returns a copy. To add a secret, use `addSecret()`.
     */
    get secrets() {
        return [...this._secrets];
    }
}
exports.ServiceAccount = ServiceAccount;
//# sourceMappingURL=data:application/json;base64,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