"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodSpecDefinition = exports.RestartPolicy = exports.Pod = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const cdk8s = require("cdk8s");
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.spec = new PodSpecDefinition(props.spec);
        this.apiObject = new k8s.Pod(this, 'Pod', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this.spec._toKube() }),
        });
    }
}
exports.Pod = Pod;
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
/**
 * A description of a pod.
 */
class PodSpecDefinition {
    constructor(props = {}) {
        var _a, _b;
        this._containers = (_a = props.containers) !== null && _a !== void 0 ? _a : [];
        this._volumes = (_b = props.volumes) !== null && _b !== void 0 ? _b : [];
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
    }
    /**
     * List of containers belonging to the pod.
     *
     * @returns a copy - do not modify
     */
    get containers() {
        return [...this._containers];
    }
    /**
     * Adds a container to this pod.
     *
     * @param container The container to add
     */
    addContainer(container) {
        this._containers.push(container);
    }
    /**
     * Adds a volume to this pod.
     *
     * @param volume The volume to add
     */
    addVolume(volume) {
        this._volumes.push(volume);
    }
    /**
     * List of volumes that can be mounted by containers belonging to the pod.
     *
     * Returns a copy. To add volumes, use `addVolume()`.
     */
    get volumes() {
        return [...this._volumes];
    }
    /**
     * @internal
     */
    _toKube() {
        var _a;
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = [];
        const containers = [];
        for (const container of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of container.mounts) {
                volumes.push(mount.volume._toKube());
            }
            containers.push(container._toKube());
        }
        for (const volume of this._volumes) {
            volumes.push(volume._toKube());
        }
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: (_a = this.serviceAccount) === null || _a === void 0 ? void 0 : _a.name,
            containers: containers,
            volumes: volumes,
        };
    }
}
exports.PodSpecDefinition = PodSpecDefinition;
//# sourceMappingURL=data:application/json;base64,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