import { Construct } from 'constructs';
import { ResourceProps, Resource } from './base';
import * as cdk8s from 'cdk8s';
import { IServiceAccount } from './service-account';
import { Container } from './container';
import { Volume } from './volume';
/**
 * Properties for initialization of `Pod`.
 */
export interface PodProps extends ResourceProps {
    /**
     * The spec of the pod. Use `pod.spec` to apply post instantiation mutations.
     *
     * @default - An empty spec will be created.
     */
    readonly spec?: PodSpec;
}
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
export declare class Pod extends Resource {
    protected readonly apiObject: cdk8s.ApiObject;
    /**
     * Provides access to the underlying spec.
     *
     * You can use this field to apply post instantiation mutations
     * to the spec.
     */
    readonly spec: PodSpecDefinition;
    constructor(scope: Construct, id: string, props?: PodProps);
}
/**
 * Properties for initialization of `PodSpec`.
 */
export interface PodSpec {
    /**
     * List of containers belonging to the pod. Containers cannot currently be
     * added or removed. There must be at least one container in a Pod.
     *
     * You can add additionnal containers using `podSpec.addContainer()`
     *
     * @default - No containers. Note that a pod spec must include at least one container.
     */
    readonly containers?: Container[];
    /**
     * List of volumes that can be mounted by containers belonging to the pod.
     *
     * You can also add volumes later using `podSpec.addVolume()`
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes
     *
     * @default - No volumes.
     */
    readonly volumes?: Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/pods/pod-lifecycle/#restart-policy
     *
     * @default RestartPolicy.ALWAYS
     */
    readonly restartPolicy?: RestartPolicy;
    /**
     * A service account provides an identity for processes that run in a Pod.
     *
     * When you (a human) access the cluster (for example, using kubectl), you are
     * authenticated by the apiserver as a particular User Account (currently this
     * is usually admin, unless your cluster administrator has customized your
     * cluster). Processes in containers inside pods can also contact the
     * apiserver. When they do, they are authenticated as a particular Service
     * Account (for example, default).
     *
     * @see https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/
     *
     * @default - No service account.
     */
    readonly serviceAccount?: IServiceAccount;
}
/**
 * Restart policy for all containers within the pod.
 */
export declare enum RestartPolicy {
    /**
     * Always restart the pod after it exits.
     */
    ALWAYS = "Always",
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    ON_FAILURE = "OnFailure",
    /**
     * Never restart the pod.
     */
    NEVER = "Never"
}
/**
 * A description of a pod.
 */
export declare class PodSpecDefinition {
    /**
     * Restart policy for all containers within the pod.
     */
    readonly restartPolicy?: RestartPolicy;
    /**
     * The service account used to run this pod.
     */
    readonly serviceAccount?: IServiceAccount;
    private readonly _containers;
    private readonly _volumes;
    constructor(props?: PodSpec);
    /**
     * List of containers belonging to the pod.
     *
     * @returns a copy - do not modify
     */
    get containers(): Container[];
    /**
     * Adds a container to this pod.
     *
     * @param container The container to add
     */
    addContainer(container: Container): void;
    /**
     * Adds a volume to this pod.
     *
     * @param volume The volume to add
     */
    addVolume(volume: Volume): void;
    /**
     * List of volumes that can be mounted by containers belonging to the pod.
     *
     * Returns a copy. To add volumes, use `addVolume()`.
     */
    get volumes(): Volume[];
}
