"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobSpecDefinition = exports.Job = void 0;
const base_1 = require("./base");
const cdk8s_1 = require("cdk8s");
const cdk8s = require("cdk8s");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
/**
 * A Job creates one or more Pods and ensures that a specified number of them successfully terminate. As pods successfully complete,
 * the Job tracks the successful completions. When a specified number of successful completions is reached, the task (ie, Job) is complete.
 * Deleting a Job will clean up the Pods it created. A simple case is to create one Job object in order to reliably run one Pod to completion.
 * The Job object will start a new Pod if the first Pod fails or is deleted (for example due to a node hardware failure or a node reboot).
 * You can also use a Job to run multiple Pods in parallel.
 */
class Job extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.spec = new JobSpecDefinition(props.spec);
        this.apiObject = new k8s.Job(this, 'Default', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this.spec._toKube() }),
        });
    }
}
exports.Job = Job;
class JobSpecDefinition {
    constructor(props = {}) {
        var _a, _b;
        this.podSpecTemplate = new pod_1.PodSpecDefinition({
            restartPolicy: (_b = (_a = props.podSpecTemplate) === null || _a === void 0 ? void 0 : _a.restartPolicy) !== null && _b !== void 0 ? _b : pod_1.RestartPolicy.NEVER,
            ...props.podSpecTemplate,
        });
        this.podMetadataTemplate = new cdk8s_1.ApiObjectMetadataDefinition(props.podMetadataTemplate);
        this.ttlAfterFinished = props.ttlAfterFinished;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            template: {
                metadata: this.podMetadataTemplate.toJson(),
                spec: this.podSpecTemplate._toKube(),
            },
            ttlSecondsAfterFinished: this.ttlAfterFinished ? this.ttlAfterFinished.toSeconds() : undefined,
        };
    }
}
exports.JobSpecDefinition = JobSpecDefinition;
//# sourceMappingURL=data:application/json;base64,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