# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mdd']

package_data = \
{'': ['*']}

install_requires = \
['bdd2dfa[nx]>=1.0.0,<2.0.0',
 'dd>=0.5.5,<0.6.0',
 'funcy>=1.14,<2.0',
 'networkx[nx]>=2.5,<3.0',
 'py-aiger-bdd>=3.0.0,<4.0.0',
 'py-aiger-bv>=4.3.0,<5.0.0',
 'py-aiger>=6.1.1,<7.0.0']

setup_kwargs = {
    'name': 'mdd',
    'version': '0.3.0',
    'description': 'Python abstraction around Binary Decision Diagrams to implement Multivalued Decision Diagrams.',
    'long_description': '# Py-MDD\n\nPython abstraction around Binary Decision Diagrams to implement\nMulti-valued Decision Diagrams.\n\n[![Build Status](https://cloud.drone.io/api/badges/mvcisback/py-mdd/status.svg)](https://cloud.drone.io/mvcisback/py-mdd)\n[![docs badge](https://img.shields.io/badge/docs-docs-black)](https://mjvc.me/py-mdd)\n[![codecov](https://codecov.io/gh/mvcisback/py-mdd/branch/main/graph/badge.svg)](https://codecov.io/gh/mvcisback/py-mdd)\n[![PyPI version](https://badge.fury.io/py/mdd.svg)](https://badge.fury.io/py/mdd)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)\n\n<!-- markdown-toc start - Don\'t edit this section. Run M-x markdown-toc-refresh-toc -->\n**Table of Contents**\n\n- [About](#about)\n- [Installation](#installation)\n- [Usage](#usage)\n    - [Variables, Interfaces, and Encodings](#variables-interfaces-and-encodings)\n    - [BDD Encoding Details](#bdd-encoding-details)\n    - [Ordering `DecisionDiagram`s](#ordering-decisiondiagrams)\n    - [Converting to Directed Graph (networkx)](#converting-to-directed-graph-networkx)\n\n<!-- markdown-toc end -->\n\n\n## About\n\nMulti-valued Decision Diagrams (MDD)  are a way to represent discrete function: \n\n    f : A₁ × A₂ × … × Aₙ → B.\n\nConceptually, a MDD for `f` can be thought of as a compressed decision tree (in the form of a directed acyclc graph).\n\nFor example, if we have a function over two variables,\n\n    x ∈ {1,2,3,4,5}, y ∈ {\'a\',\'b\'}\n\nwith possible outputs `f(x, y) ∈ {-1, 0, 1}`, then the following diagram represents the function:\n\n    f(x, y) = 1 if (x ≡ 1 and y ≡ \'a\') else 0\n\n<figure style="display: flex; justify-content: center; flex-direction: row">\n<img src="assets/example_mdd.png"\n</figure>\n    \nThis library provides abstractions to easily create and manipulate MDDs.\n\n# Installation\n\nIf you just need to use `py-mdd`, you can just run:\n\n`$ pip install mdd`\n\nFor developers, note that this project uses the\n[poetry](https://poetry.eustace.io/) python package/dependency\nmanagement tool. Please familarize yourself with it and then\nrun:\n\n`$ poetry install`\n\n\n# Usage\n\nFor the impatient, here is a basic usage example:\n\n```python\nimport mdd\n\ninterface = mdd.Interface(\n    inputs={\n        "x": [1, 2, 3],\n        "y": [6, \'w\'], \n        "z": [7, True, 8],\n    }, \n    output=[-1, 0, 1],\n)\nfunc = interface.constantly(-1)\nassert func({\'x\': 1, \'y\': \'w\', \'z\': 8}) == -1\n\n# Can access underlying BDD from `dd` library.\n# Note: This BDD encodes both the function\'s output\n#       *and* domain (valid inputs).\nassert func.bdd.dag_size == 33\n```\n\nIf 33 seems very large to you, this is just a constant function after\nall, note that as the following sections illustrate, its easy to\nimplement alternative encodings which can be much more compact. <a href="#note2">[0]</a>\n\n## Variables, Interfaces, and Encodings\n\nThe `mdd` api centers around three objects:\n\n1. [`Variable`](https://mjvc.me/py-mdd/mdd.html#mdd.mdd.Variable): Representation of a named variable taking on values in\n   from a finite set described by an aiger circuit.\n1. [`Interface`](https://mjvc.me/py-mdd/mdd.html#mdd.mdd.Interface): Description of inputs and outputs of a Multi-valued Decision Diagram.\n1. [`DecisionDiagram`](https://mjvc.me/py-mdd/mdd.html#mdd.mdd.DecisionDiagram): Representation of a Multi-valued Decision Diagram that conforms\n   to an interface. This object is a wrapper around a Binary Decision Diagram object (from\n[dd](https://github.com/tulip-control/dd)).\n\nBy default, variables use one-hot encoding, but all **input**\nvariables can use arbitrary encodings by defining a bit-vector\nexpression describing **valid** inputs and a encode/decoder pair from\n`int`s to the variable\'s domain.\n\n\n```python\n# One hot encoded by default.\nvar1 = mdd.to_var(domain=["x", "y", "z"], name="myvar1")\n\n# Hand crafted encoding using `py-aiger`.\n\nimport aiger_bv\n\n# Named 2-length bitvector circuit.\nbvexpr = aiger_bv.uatom(2, \'myvar3\')\n\ndomain = (\'a\', \'b\', \'c\')\nvar2 = mdd.Variable(\n     encode=domain.index,        # Any -> int\n     decode=domain.__getitem__,  # int -> Any\n     valid=bvexpr < 4,           # 0b11 is invalid!\n)\n\n# Can create new variable using same encoding, but different name.\nvar3 = var2.with_name("myvar3")\n\nvar4 = mdd.to_var(domain=[-1, 0, 1], name=\'output\')\n```\n\nA useful feature of variables is that they can generate an `aiger_bv`\nBitVector object to describe circuits in terms of a variable.\n\n```python\na_int = var2.encode(\'a\')\ny_int = var1.encode(\'y\')\n\n# BitVector Expression testing if var2 is \'a\' and var1 is \'y\'.\nexpr = (var2.expr() == a_int) & (var1.expr() == y_int)\n```\n\nGiven these variables, we can define an input/output interface.\n\n```python\n# All variables must have distinct names.\ninterface = mdd.Interface(inputs=[var1, var2, var3], output=var4)\n```\n\nFurther, as the first example showed, if the default encoding is fine,\nthen we can simply pass a dictionary inplace of inputs and/or a\niterable in place of the output. In this case, a 1-hot encoding will\nbe created using the order of the variables.\n\n```python\ninterface = mdd.Interface(\n    inputs={\n        "x": [1, 2, 3],      # These are\n        "y": [6, \'w\'],       # 1-hot\n        "z": [7, True, 8],   # Encoded.\n    }, \n    output=[-1, 0, 1],       # uuid based output name.\n)\n```\n\nFinally, given an interface we can create a Multi-valued Decision\nDiagram. There are five main ways to create a `DecisionDiagram`:\n\n1. Given an interface, create a constant function:\n   ```python\n   func = interface.constantly(1)\n   ```\n\n2. Wrap an `py-aiger` compatible object:\n   ```python\n   import aiger_bv as BV\n    \n   x = interface.var(\'x\')  # Access \'x\' variable.\n   out = interface.output  # Access output variable.\n\n   expr = BV.ite(\n       x.expr() == x.encode(2),       # Test.\n       out.expr() == out.encode(0),   # True branch.\n       out.expr() == out.encode(-1),  # False branch.\n   )\n   func = interface.lift(expr)\n   assert func({\'x\': 2, \'y\': 6, \'z\': True}) == 0\n   assert func({\'x\': 1, \'y\': 6, \'z\': True}) == -1\n   ```\n\n3. Wrap an existing Binary Decision Diagram:\n   ```python\n   bdd = mdd.to_bdd(expr)      # Convert `aiger` expression to bdd.\n   func = interface.lift(bdd)  # bdd type comes from `dd` library.\n   assert func({\'x\': 2, \'y\': 6, \'z\': True}) == 0\n   assert func({\'x\': 1, \'y\': 6, \'z\': True}) == -1\n   ```\n\n4. Partially Evaluate an existing `DecisionDiagram`:\n   ```python\n   constantly_0 = func.let({\'x\': 2})\n   assert func({\'y\': 6, \'z\': True}) == 0\n   ```\n\n5. Override an existing `DecisionDiagram` given a predicate:\n   ```python\n   # Can be a BDD or and py-aiger compatible object.\n   test = x.expr() == x.encode(1)\n   # If x = 1, then return 1, otherwise return using func.\n   func2 = func.override(test=test, value=1)\n   assert func2({\'x\': 1, \'y\': 6, \'z\': True}) == 1\n   ```\n\n## BDD Encoding Details\n\nThe `py-mdd` library uses a Binary Decision Diagram to represent a\nmulti-valued function. The encoding slighly differs from the standard\nreduction <a href="#note1">[1]</a> from mdds to bdds by assuming the following:\n\n1. If a variable encoding is invalid, then the bdd maps it to `0`.\n1. The output is 1-hot encoded, i.e., there is a variable for each\n   outcome.\n1. If a `bdd` has all input variables fixed to a valid assignment, the\n   resulting `bdd` depends on exactly one output varible, which then\n   determines the output.\n\nAny bdd that conforms to this encoding can be wrapped up by an\napproriate `Interface`.\n\n## Ordering `DecisionDiagram`s\n\nThe underlying BDD can be reordered to respect variable ordering by\nproviding a complete list of variable names to the `order` method.\n\n```python\nfunc.order([\'x\', \'y\', \'z\', func.output.name])\n```\n\n## Converting to Directed Graph (networkx)\n\nIf the `networkx` python package is installed:\n\n`$ pip install networkx`\n\nor the `nx` option is added when installing `py-mdd`:\n\n`$ pip install mdd[nx]`\n\nthen one can export a `DecisionDiagram` as a directed graph:\n\n```python\nfrom mdd.nx import to_nx\n\ngraph = to_nx(func)  # Has BitVector expressions on edges to represent guards.\n\ngraph2 = to_nx(func, symbolic_edges=False)  # Has explicit sets of values on edges to represent guards.\n```\n\n\n<strong id="note2">[0]:</strong> To get a sense for how much overhead is introduced, consider the corresponding Zero Suppressed Decision Diagram (ZDD) of a 1-hot encoding. A classic result (see Art of Computer Programming vol 4a) is the ZDD size bounds the BDD size via O(#variables*|size of ZDD|).\n\n<strong id="note1">[1]:</strong> Srinivasan, Arvind, et al. "Algorithms for discrete function manipulation." 1990 IEEE international conference on computer-aided design. IEEE Computer Society, 1990.\n\n',
    'author': 'Marcell Vazquez-Chanlatte',
    'author_email': 'mvc@linux.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mvcisback/py-mdd',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
