"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const db = require("@aws-cdk/aws-dynamodb");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const ObjectType = require("./object-type-definitions");
const ScalarType = require("./scalar-type-defintions");
/*
 * Creates an Appsync GraphQL API and schema in a code-first approach.
 *
 * Stack verification steps:
 * Deploy stack, get api key and endpoinScalarType. Check if schema connects to data source.
 *
 * -- bash verify.integ.graphql-schema.sh --start                 -- start                    --
 * -- aws appsync list-graphql-apis                               -- obtain apiId & endpoint  --
 * -- aws appsync list-api-keys --api-id [apiId]                  -- obtain api key           --
 * -- bash verify.integ.graphql-schema.sh --check [apiKey] [url]  -- check if success         --
 * -- bash verify.integ.graphql-schema.sh --clean                 -- clean                    --
 */
const app = new cdk.App();
const stack = new cdk.Stack(app, 'code-first-schema');
const schema = new appsync.Schema();
const node = new appsync.InterfaceType('Node', {
    definition: {
        created: ScalarType.string,
        edited: ScalarType.string,
        id: ScalarType.required_id,
    },
});
schema.addType(node);
const api = new appsync.GraphqlApi(stack, 'code-first-api', {
    name: 'api',
    schema: schema,
});
const table = new db.Table(stack, 'table', {
    partitionKey: {
        name: 'id',
        type: db.AttributeType.STRING,
    },
});
const tableDS = api.addDynamoDbDataSource('planets', table);
const planet = ObjectType.planet;
schema.addType(planet);
const species = api.addType(new appsync.ObjectType('Species', {
    interfaceTypes: [node],
    definition: {
        name: ScalarType.string,
        classification: ScalarType.string,
        designation: ScalarType.string,
        averageHeight: ScalarType.float,
        averageLifespan: ScalarType.int,
        eyeColors: ScalarType.list_string,
        hairColors: ScalarType.list_string,
        skinColors: ScalarType.list_string,
        language: ScalarType.string,
        homeworld: planet.attribute(),
    },
}));
api.addQuery('getPlanets', new appsync.ResolvableField({
    returnType: planet.attribute({ isList: true }),
    dataSource: tableDS,
    requestMappingTemplate: appsync.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: appsync.MappingTemplate.dynamoDbResultList(),
}));
/* ATTRIBUTES */
const name = new appsync.Assign('name', '$context.arguments.name');
const diameter = new appsync.Assign('diameter', '$context.arguments.diameter');
const rotationPeriod = new appsync.Assign('rotationPeriod', '$context.arguments.rotationPeriod');
const orbitalPeriod = new appsync.Assign('orbitalPeriod', '$context.arguments.orbitalPeriod');
const gravity = new appsync.Assign('gravityPeriod', '$context.arguments.gravity');
const population = new appsync.Assign('population', '$context.arguments.population');
const climates = new appsync.Assign('climates', '$context.arguments.climates');
const terrains = new appsync.Assign('terrains', '$context.arguments.terrains');
const surfaceWater = new appsync.Assign('surfaceWater', '$context.arguments.surfaceWater');
api.addMutation('addPlanet', new appsync.ResolvableField({
    returnType: planet.attribute(),
    args: {
        name: ScalarType.string,
        diameter: ScalarType.int,
        rotationPeriod: ScalarType.int,
        orbitalPeriod: ScalarType.int,
        gravity: ScalarType.string,
        population: ScalarType.list_string,
        climates: ScalarType.list_string,
        terrains: ScalarType.list_string,
        surfaceWater: ScalarType.float,
    },
    dataSource: tableDS,
    requestMappingTemplate: appsync.MappingTemplate.dynamoDbPutItem(appsync.PrimaryKey.partition('id').auto(), new appsync.AttributeValues('$context.arguments', [name, diameter, rotationPeriod, orbitalPeriod, gravity, population, climates, terrains, surfaceWater])),
    responseMappingTemplate: appsync.MappingTemplate.dynamoDbResultItem(),
}));
api.addSubscription('addedPlanets', new appsync.ResolvableField({
    returnType: planet.attribute(),
    args: { id: ScalarType.required_id },
    directives: [appsync.Directive.subscribe('addPlanet')],
}));
api.addType(new appsync.InputType('AwesomeInput', {
    definition: { awesomeInput: ScalarType.string },
}));
api.addType(new appsync.EnumType('Episodes', {
    definition: [
        'The_Phantom_Menace',
        'Attack_of_the_Clones',
        'Revenge_of_the_Sith',
        'A_New_Hope',
        'The_Empire_Strikes_Back',
        'Return_of_the_Jedi',
        'The_Force_Awakens',
        'The_Last_Jedi',
        'The_Rise_of_Skywalker',
    ],
}));
api.addType(new appsync.UnionType('Union', {
    definition: [species, planet],
}));
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuZ3JhcGhxbC1zY2hlbWEuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnRlZy5ncmFwaHFsLXNjaGVtYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLDRDQUE0QztBQUM1QyxxQ0FBcUM7QUFDckMsa0NBQWtDO0FBQ2xDLHdEQUF3RDtBQUN4RCx1REFBdUQ7QUFFdkQ7Ozs7Ozs7Ozs7O0dBV0c7QUFDSCxNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQztBQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLG1CQUFtQixDQUFDLENBQUM7QUFFdEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxPQUFPLENBQUMsTUFBTSxFQUFFLENBQUM7QUFFcEMsTUFBTSxJQUFJLEdBQUcsSUFBSSxPQUFPLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRTtJQUM3QyxVQUFVLEVBQUU7UUFDVixPQUFPLEVBQUUsVUFBVSxDQUFDLE1BQU07UUFDMUIsTUFBTSxFQUFFLFVBQVUsQ0FBQyxNQUFNO1FBQ3pCLEVBQUUsRUFBRSxVQUFVLENBQUMsV0FBVztLQUMzQjtDQUNGLENBQUMsQ0FBQztBQUVILE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7QUFFckIsTUFBTSxHQUFHLEdBQUcsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtJQUMxRCxJQUFJLEVBQUUsS0FBSztJQUNYLE1BQU0sRUFBRSxNQUFNO0NBQ2YsQ0FBQyxDQUFDO0FBRUgsTUFBTSxLQUFLLEdBQUcsSUFBSSxFQUFFLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7SUFDekMsWUFBWSxFQUFFO1FBQ1osSUFBSSxFQUFFLElBQUk7UUFDVixJQUFJLEVBQUUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxNQUFNO0tBQzlCO0NBQ0YsQ0FBQyxDQUFDO0FBRUgsTUFBTSxPQUFPLEdBQUcsR0FBRyxDQUFDLHFCQUFxQixDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUU1RCxNQUFNLE1BQU0sR0FBRyxVQUFVLENBQUMsTUFBTSxDQUFDO0FBQ2pDLE1BQU0sQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7QUFFdkIsTUFBTSxPQUFPLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsU0FBUyxFQUFFO0lBQzVELGNBQWMsRUFBRSxDQUFDLElBQUksQ0FBQztJQUN0QixVQUFVLEVBQUU7UUFDVixJQUFJLEVBQUUsVUFBVSxDQUFDLE1BQU07UUFDdkIsY0FBYyxFQUFFLFVBQVUsQ0FBQyxNQUFNO1FBQ2pDLFdBQVcsRUFBRSxVQUFVLENBQUMsTUFBTTtRQUM5QixhQUFhLEVBQUUsVUFBVSxDQUFDLEtBQUs7UUFDL0IsZUFBZSxFQUFFLFVBQVUsQ0FBQyxHQUFHO1FBQy9CLFNBQVMsRUFBRSxVQUFVLENBQUMsV0FBVztRQUNqQyxVQUFVLEVBQUUsVUFBVSxDQUFDLFdBQVc7UUFDbEMsVUFBVSxFQUFFLFVBQVUsQ0FBQyxXQUFXO1FBQ2xDLFFBQVEsRUFBRSxVQUFVLENBQUMsTUFBTTtRQUMzQixTQUFTLEVBQUUsTUFBTSxDQUFDLFNBQVMsRUFBRTtLQUM5QjtDQUNGLENBQUMsQ0FBQyxDQUFDO0FBRUosR0FBRyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUUsSUFBSSxPQUFPLENBQUMsZUFBZSxDQUFDO0lBQ3JELFVBQVUsRUFBRSxNQUFNLENBQUMsU0FBUyxDQUFDLEVBQUUsTUFBTSxFQUFFLElBQUksRUFBRSxDQUFDO0lBQzlDLFVBQVUsRUFBRSxPQUFPO0lBQ25CLHNCQUFzQixFQUFFLE9BQU8sQ0FBQyxlQUFlLENBQUMsaUJBQWlCLEVBQUU7SUFDbkUsdUJBQXVCLEVBQUUsT0FBTyxDQUFDLGVBQWUsQ0FBQyxrQkFBa0IsRUFBRTtDQUN0RSxDQUFDLENBQUMsQ0FBQztBQUVKLGdCQUFnQjtBQUNoQixNQUFNLElBQUksR0FBRyxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLHlCQUF5QixDQUFDLENBQUM7QUFDbkUsTUFBTSxRQUFRLEdBQUcsSUFBSSxPQUFPLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRSw2QkFBNkIsQ0FBQyxDQUFDO0FBQy9FLE1BQU0sY0FBYyxHQUFHLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsRUFBRSxtQ0FBbUMsQ0FBQyxDQUFDO0FBQ2pHLE1BQU0sYUFBYSxHQUFHLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQyxlQUFlLEVBQUUsa0NBQWtDLENBQUMsQ0FBQztBQUM5RixNQUFNLE9BQU8sR0FBRyxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUMsZUFBZSxFQUFFLDRCQUE0QixDQUFDLENBQUM7QUFDbEYsTUFBTSxVQUFVLEdBQUcsSUFBSSxPQUFPLENBQUMsTUFBTSxDQUFDLFlBQVksRUFBRSwrQkFBK0IsQ0FBQyxDQUFDO0FBQ3JGLE1BQU0sUUFBUSxHQUFHLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQyxVQUFVLEVBQUUsNkJBQTZCLENBQUMsQ0FBQztBQUMvRSxNQUFNLFFBQVEsR0FBRyxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFLDZCQUE2QixDQUFDLENBQUM7QUFDL0UsTUFBTSxZQUFZLEdBQUcsSUFBSSxPQUFPLENBQUMsTUFBTSxDQUFDLGNBQWMsRUFBRSxpQ0FBaUMsQ0FBQyxDQUFDO0FBQzNGLEdBQUcsQ0FBQyxXQUFXLENBQUMsV0FBVyxFQUFFLElBQUksT0FBTyxDQUFDLGVBQWUsQ0FBQztJQUN2RCxVQUFVLEVBQUUsTUFBTSxDQUFDLFNBQVMsRUFBRTtJQUM5QixJQUFJLEVBQUU7UUFDSixJQUFJLEVBQUUsVUFBVSxDQUFDLE1BQU07UUFDdkIsUUFBUSxFQUFFLFVBQVUsQ0FBQyxHQUFHO1FBQ3hCLGNBQWMsRUFBRSxVQUFVLENBQUMsR0FBRztRQUM5QixhQUFhLEVBQUUsVUFBVSxDQUFDLEdBQUc7UUFDN0IsT0FBTyxFQUFFLFVBQVUsQ0FBQyxNQUFNO1FBQzFCLFVBQVUsRUFBRSxVQUFVLENBQUMsV0FBVztRQUNsQyxRQUFRLEVBQUUsVUFBVSxDQUFDLFdBQVc7UUFDaEMsUUFBUSxFQUFFLFVBQVUsQ0FBQyxXQUFXO1FBQ2hDLFlBQVksRUFBRSxVQUFVLENBQUMsS0FBSztLQUMvQjtJQUNELFVBQVUsRUFBRSxPQUFPO0lBQ25CLHNCQUFzQixFQUFFLE9BQU8sQ0FBQyxlQUFlLENBQUMsZUFBZSxDQUM3RCxPQUFPLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxJQUFJLE9BQU8sQ0FBQyxlQUFlLENBQUMsb0JBQW9CLEVBQ3pGLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRSxjQUFjLEVBQUUsYUFBYSxFQUFFLE9BQU8sRUFBRSxVQUFVLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxZQUFZLENBQUMsQ0FDdkcsQ0FDRjtJQUNELHVCQUF1QixFQUFFLE9BQU8sQ0FBQyxlQUFlLENBQUMsa0JBQWtCLEVBQUU7Q0FDdEUsQ0FBQyxDQUFDLENBQUM7QUFFSixHQUFHLENBQUMsZUFBZSxDQUFDLGNBQWMsRUFBRSxJQUFJLE9BQU8sQ0FBQyxlQUFlLENBQUM7SUFDOUQsVUFBVSxFQUFFLE1BQU0sQ0FBQyxTQUFTLEVBQUU7SUFDOUIsSUFBSSxFQUFFLEVBQUUsRUFBRSxFQUFFLFVBQVUsQ0FBQyxXQUFXLEVBQUU7SUFDcEMsVUFBVSxFQUFFLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLENBQUM7Q0FDdkQsQ0FBQyxDQUFDLENBQUM7QUFDSixHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksT0FBTyxDQUFDLFNBQVMsQ0FBQyxjQUFjLEVBQUU7SUFDaEQsVUFBVSxFQUFFLEVBQUUsWUFBWSxFQUFFLFVBQVUsQ0FBQyxNQUFNLEVBQUU7Q0FDaEQsQ0FBQyxDQUFDLENBQUM7QUFFSixHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxVQUFVLEVBQUU7SUFDM0MsVUFBVSxFQUFFO1FBQ1Ysb0JBQW9CO1FBQ3BCLHNCQUFzQjtRQUN0QixxQkFBcUI7UUFDckIsWUFBWTtRQUNaLHlCQUF5QjtRQUN6QixvQkFBb0I7UUFDcEIsbUJBQW1CO1FBQ25CLGVBQWU7UUFDZix1QkFBdUI7S0FDeEI7Q0FDRixDQUFDLENBQUMsQ0FBQztBQUVKLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxPQUFPLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRTtJQUN6QyxVQUFVLEVBQUUsQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDO0NBQzlCLENBQUMsQ0FBQyxDQUFDO0FBRUosR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZGIgZnJvbSAnQGF3cy1jZGsvYXdzLWR5bmFtb2RiJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGFwcHN5bmMgZnJvbSAnLi4vbGliJztcbmltcG9ydCAqIGFzIE9iamVjdFR5cGUgZnJvbSAnLi9vYmplY3QtdHlwZS1kZWZpbml0aW9ucyc7XG5pbXBvcnQgKiBhcyBTY2FsYXJUeXBlIGZyb20gJy4vc2NhbGFyLXR5cGUtZGVmaW50aW9ucyc7XG5cbi8qXG4gKiBDcmVhdGVzIGFuIEFwcHN5bmMgR3JhcGhRTCBBUEkgYW5kIHNjaGVtYSBpbiBhIGNvZGUtZmlyc3QgYXBwcm9hY2guXG4gKlxuICogU3RhY2sgdmVyaWZpY2F0aW9uIHN0ZXBzOlxuICogRGVwbG95IHN0YWNrLCBnZXQgYXBpIGtleSBhbmQgZW5kcG9pblNjYWxhclR5cGUuIENoZWNrIGlmIHNjaGVtYSBjb25uZWN0cyB0byBkYXRhIHNvdXJjZS5cbiAqXG4gKiAtLSBiYXNoIHZlcmlmeS5pbnRlZy5ncmFwaHFsLXNjaGVtYS5zaCAtLXN0YXJ0ICAgICAgICAgICAgICAgICAtLSBzdGFydCAgICAgICAgICAgICAgICAgICAgLS1cbiAqIC0tIGF3cyBhcHBzeW5jIGxpc3QtZ3JhcGhxbC1hcGlzICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIC0tIG9idGFpbiBhcGlJZCAmIGVuZHBvaW50ICAtLVxuICogLS0gYXdzIGFwcHN5bmMgbGlzdC1hcGkta2V5cyAtLWFwaS1pZCBbYXBpSWRdICAgICAgICAgICAgICAgICAgLS0gb2J0YWluIGFwaSBrZXkgICAgICAgICAgIC0tXG4gKiAtLSBiYXNoIHZlcmlmeS5pbnRlZy5ncmFwaHFsLXNjaGVtYS5zaCAtLWNoZWNrIFthcGlLZXldIFt1cmxdICAtLSBjaGVjayBpZiBzdWNjZXNzICAgICAgICAgLS1cbiAqIC0tIGJhc2ggdmVyaWZ5LmludGVnLmdyYXBocWwtc2NoZW1hLnNoIC0tY2xlYW4gICAgICAgICAgICAgICAgIC0tIGNsZWFuICAgICAgICAgICAgICAgICAgICAtLVxuICovXG5jb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKGFwcCwgJ2NvZGUtZmlyc3Qtc2NoZW1hJyk7XG5cbmNvbnN0IHNjaGVtYSA9IG5ldyBhcHBzeW5jLlNjaGVtYSgpO1xuXG5jb25zdCBub2RlID0gbmV3IGFwcHN5bmMuSW50ZXJmYWNlVHlwZSgnTm9kZScsIHtcbiAgZGVmaW5pdGlvbjoge1xuICAgIGNyZWF0ZWQ6IFNjYWxhclR5cGUuc3RyaW5nLFxuICAgIGVkaXRlZDogU2NhbGFyVHlwZS5zdHJpbmcsXG4gICAgaWQ6IFNjYWxhclR5cGUucmVxdWlyZWRfaWQsXG4gIH0sXG59KTtcblxuc2NoZW1hLmFkZFR5cGUobm9kZSk7XG5cbmNvbnN0IGFwaSA9IG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdjb2RlLWZpcnN0LWFwaScsIHtcbiAgbmFtZTogJ2FwaScsXG4gIHNjaGVtYTogc2NoZW1hLFxufSk7XG5cbmNvbnN0IHRhYmxlID0gbmV3IGRiLlRhYmxlKHN0YWNrLCAndGFibGUnLCB7XG4gIHBhcnRpdGlvbktleToge1xuICAgIG5hbWU6ICdpZCcsXG4gICAgdHlwZTogZGIuQXR0cmlidXRlVHlwZS5TVFJJTkcsXG4gIH0sXG59KTtcblxuY29uc3QgdGFibGVEUyA9IGFwaS5hZGREeW5hbW9EYkRhdGFTb3VyY2UoJ3BsYW5ldHMnLCB0YWJsZSk7XG5cbmNvbnN0IHBsYW5ldCA9IE9iamVjdFR5cGUucGxhbmV0O1xuc2NoZW1hLmFkZFR5cGUocGxhbmV0KTtcblxuY29uc3Qgc3BlY2llcyA9IGFwaS5hZGRUeXBlKG5ldyBhcHBzeW5jLk9iamVjdFR5cGUoJ1NwZWNpZXMnLCB7XG4gIGludGVyZmFjZVR5cGVzOiBbbm9kZV0sXG4gIGRlZmluaXRpb246IHtcbiAgICBuYW1lOiBTY2FsYXJUeXBlLnN0cmluZyxcbiAgICBjbGFzc2lmaWNhdGlvbjogU2NhbGFyVHlwZS5zdHJpbmcsXG4gICAgZGVzaWduYXRpb246IFNjYWxhclR5cGUuc3RyaW5nLFxuICAgIGF2ZXJhZ2VIZWlnaHQ6IFNjYWxhclR5cGUuZmxvYXQsXG4gICAgYXZlcmFnZUxpZmVzcGFuOiBTY2FsYXJUeXBlLmludCxcbiAgICBleWVDb2xvcnM6IFNjYWxhclR5cGUubGlzdF9zdHJpbmcsXG4gICAgaGFpckNvbG9yczogU2NhbGFyVHlwZS5saXN0X3N0cmluZyxcbiAgICBza2luQ29sb3JzOiBTY2FsYXJUeXBlLmxpc3Rfc3RyaW5nLFxuICAgIGxhbmd1YWdlOiBTY2FsYXJUeXBlLnN0cmluZyxcbiAgICBob21ld29ybGQ6IHBsYW5ldC5hdHRyaWJ1dGUoKSxcbiAgfSxcbn0pKTtcblxuYXBpLmFkZFF1ZXJ5KCdnZXRQbGFuZXRzJywgbmV3IGFwcHN5bmMuUmVzb2x2YWJsZUZpZWxkKHtcbiAgcmV0dXJuVHlwZTogcGxhbmV0LmF0dHJpYnV0ZSh7IGlzTGlzdDogdHJ1ZSB9KSxcbiAgZGF0YVNvdXJjZTogdGFibGVEUyxcbiAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogYXBwc3luYy5NYXBwaW5nVGVtcGxhdGUuZHluYW1vRGJTY2FuVGFibGUoKSxcbiAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IGFwcHN5bmMuTWFwcGluZ1RlbXBsYXRlLmR5bmFtb0RiUmVzdWx0TGlzdCgpLFxufSkpO1xuXG4vKiBBVFRSSUJVVEVTICovXG5jb25zdCBuYW1lID0gbmV3IGFwcHN5bmMuQXNzaWduKCduYW1lJywgJyRjb250ZXh0LmFyZ3VtZW50cy5uYW1lJyk7XG5jb25zdCBkaWFtZXRlciA9IG5ldyBhcHBzeW5jLkFzc2lnbignZGlhbWV0ZXInLCAnJGNvbnRleHQuYXJndW1lbnRzLmRpYW1ldGVyJyk7XG5jb25zdCByb3RhdGlvblBlcmlvZCA9IG5ldyBhcHBzeW5jLkFzc2lnbigncm90YXRpb25QZXJpb2QnLCAnJGNvbnRleHQuYXJndW1lbnRzLnJvdGF0aW9uUGVyaW9kJyk7XG5jb25zdCBvcmJpdGFsUGVyaW9kID0gbmV3IGFwcHN5bmMuQXNzaWduKCdvcmJpdGFsUGVyaW9kJywgJyRjb250ZXh0LmFyZ3VtZW50cy5vcmJpdGFsUGVyaW9kJyk7XG5jb25zdCBncmF2aXR5ID0gbmV3IGFwcHN5bmMuQXNzaWduKCdncmF2aXR5UGVyaW9kJywgJyRjb250ZXh0LmFyZ3VtZW50cy5ncmF2aXR5Jyk7XG5jb25zdCBwb3B1bGF0aW9uID0gbmV3IGFwcHN5bmMuQXNzaWduKCdwb3B1bGF0aW9uJywgJyRjb250ZXh0LmFyZ3VtZW50cy5wb3B1bGF0aW9uJyk7XG5jb25zdCBjbGltYXRlcyA9IG5ldyBhcHBzeW5jLkFzc2lnbignY2xpbWF0ZXMnLCAnJGNvbnRleHQuYXJndW1lbnRzLmNsaW1hdGVzJyk7XG5jb25zdCB0ZXJyYWlucyA9IG5ldyBhcHBzeW5jLkFzc2lnbigndGVycmFpbnMnLCAnJGNvbnRleHQuYXJndW1lbnRzLnRlcnJhaW5zJyk7XG5jb25zdCBzdXJmYWNlV2F0ZXIgPSBuZXcgYXBwc3luYy5Bc3NpZ24oJ3N1cmZhY2VXYXRlcicsICckY29udGV4dC5hcmd1bWVudHMuc3VyZmFjZVdhdGVyJyk7XG5hcGkuYWRkTXV0YXRpb24oJ2FkZFBsYW5ldCcsIG5ldyBhcHBzeW5jLlJlc29sdmFibGVGaWVsZCh7XG4gIHJldHVyblR5cGU6IHBsYW5ldC5hdHRyaWJ1dGUoKSxcbiAgYXJnczoge1xuICAgIG5hbWU6IFNjYWxhclR5cGUuc3RyaW5nLFxuICAgIGRpYW1ldGVyOiBTY2FsYXJUeXBlLmludCxcbiAgICByb3RhdGlvblBlcmlvZDogU2NhbGFyVHlwZS5pbnQsXG4gICAgb3JiaXRhbFBlcmlvZDogU2NhbGFyVHlwZS5pbnQsXG4gICAgZ3Jhdml0eTogU2NhbGFyVHlwZS5zdHJpbmcsXG4gICAgcG9wdWxhdGlvbjogU2NhbGFyVHlwZS5saXN0X3N0cmluZyxcbiAgICBjbGltYXRlczogU2NhbGFyVHlwZS5saXN0X3N0cmluZyxcbiAgICB0ZXJyYWluczogU2NhbGFyVHlwZS5saXN0X3N0cmluZyxcbiAgICBzdXJmYWNlV2F0ZXI6IFNjYWxhclR5cGUuZmxvYXQsXG4gIH0sXG4gIGRhdGFTb3VyY2U6IHRhYmxlRFMsXG4gIHJlcXVlc3RNYXBwaW5nVGVtcGxhdGU6IGFwcHN5bmMuTWFwcGluZ1RlbXBsYXRlLmR5bmFtb0RiUHV0SXRlbShcbiAgICBhcHBzeW5jLlByaW1hcnlLZXkucGFydGl0aW9uKCdpZCcpLmF1dG8oKSwgbmV3IGFwcHN5bmMuQXR0cmlidXRlVmFsdWVzKCckY29udGV4dC5hcmd1bWVudHMnLFxuICAgICAgW25hbWUsIGRpYW1ldGVyLCByb3RhdGlvblBlcmlvZCwgb3JiaXRhbFBlcmlvZCwgZ3Jhdml0eSwgcG9wdWxhdGlvbiwgY2xpbWF0ZXMsIHRlcnJhaW5zLCBzdXJmYWNlV2F0ZXJdLFxuICAgICksXG4gICksXG4gIHJlc3BvbnNlTWFwcGluZ1RlbXBsYXRlOiBhcHBzeW5jLk1hcHBpbmdUZW1wbGF0ZS5keW5hbW9EYlJlc3VsdEl0ZW0oKSxcbn0pKTtcblxuYXBpLmFkZFN1YnNjcmlwdGlvbignYWRkZWRQbGFuZXRzJywgbmV3IGFwcHN5bmMuUmVzb2x2YWJsZUZpZWxkKHtcbiAgcmV0dXJuVHlwZTogcGxhbmV0LmF0dHJpYnV0ZSgpLFxuICBhcmdzOiB7IGlkOiBTY2FsYXJUeXBlLnJlcXVpcmVkX2lkIH0sXG4gIGRpcmVjdGl2ZXM6IFthcHBzeW5jLkRpcmVjdGl2ZS5zdWJzY3JpYmUoJ2FkZFBsYW5ldCcpXSxcbn0pKTtcbmFwaS5hZGRUeXBlKG5ldyBhcHBzeW5jLklucHV0VHlwZSgnQXdlc29tZUlucHV0Jywge1xuICBkZWZpbml0aW9uOiB7IGF3ZXNvbWVJbnB1dDogU2NhbGFyVHlwZS5zdHJpbmcgfSxcbn0pKTtcblxuYXBpLmFkZFR5cGUobmV3IGFwcHN5bmMuRW51bVR5cGUoJ0VwaXNvZGVzJywge1xuICBkZWZpbml0aW9uOiBbXG4gICAgJ1RoZV9QaGFudG9tX01lbmFjZScsXG4gICAgJ0F0dGFja19vZl90aGVfQ2xvbmVzJyxcbiAgICAnUmV2ZW5nZV9vZl90aGVfU2l0aCcsXG4gICAgJ0FfTmV3X0hvcGUnLFxuICAgICdUaGVfRW1waXJlX1N0cmlrZXNfQmFjaycsXG4gICAgJ1JldHVybl9vZl90aGVfSmVkaScsXG4gICAgJ1RoZV9Gb3JjZV9Bd2FrZW5zJyxcbiAgICAnVGhlX0xhc3RfSmVkaScsXG4gICAgJ1RoZV9SaXNlX29mX1NreXdhbGtlcicsXG4gIF0sXG59KSk7XG5cbmFwaS5hZGRUeXBlKG5ldyBhcHBzeW5jLlVuaW9uVHlwZSgnVW5pb24nLCB7XG4gIGRlZmluaXRpb246IFtzcGVjaWVzLCBwbGFuZXRdLFxufSkpO1xuXG5hcHAuc3ludGgoKTsiXX0=