"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/*
 * Creates an Appsync GraphQL API with API_KEY authorization.
 * Testing for API_KEY Authorization.
 *
 * Stack verification steps:
 * Deploy stack, get api-key and endpoint.
 * Check if authorization occurs with empty get.
 *
 * -- bash verify.integ.auth-apikey.sh --start                      -- deploy stack               --
 * -- aws appsync list-graphql-apis                                 -- obtain api id && endpoint  --
 * -- aws appsync list-api-keys --api-id [API ID]                   -- obtain api key             --
 * -- bash verify.integ.auth-apikey.sh --check [APIKEY] [ENDPOINT]  -- check if fails/success     --
 * -- bash verify.integ.auth-apikey.sh --clean                      -- clean dependencies/stack   --
 */
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const api = new lib_1.GraphqlApi(stack, 'Api', {
    name: 'Integ_Test_APIKey',
    schema: lib_1.Schema.fromAsset(path_1.join(__dirname, 'appsync.auth.graphql')),
    authorizationConfig: {
        defaultAuthorization: {
            authorizationType: lib_1.AuthorizationType.API_KEY,
            apiKeyConfig: {
                // Generate a timestamp that's 365 days ahead, use atTimestamp so integ test doesn't fail
                expires: core_1.Expiration.atTimestamp(1626566400000),
            },
        },
    },
});
const testTable = new aws_dynamodb_1.Table(stack, 'TestTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const testDS = api.addDynamoDbDataSource('testDataSource', testTable);
testDS.createResolver({
    typeName: 'Query',
    fieldName: 'getTests',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
testDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addTest',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('test')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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