"use strict";
/// !cdk-integ *
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const db = require("@aws-cdk/aws-dynamodb");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
/*
 * Creates an Appsync GraphQL API in a separate stack.
 * Add dependencies to imported api.
 *
 * Stack verification steps:
 * Install dependencies and deploy integration test. Check if data sources are
 * connected to the GraphQL Api
 *
 * -- cdk deploy --app 'node integ.api-import.js' stack            -- start         --
 * -- aws appsync list-graphql-apis                                -- obtain api id --
 * -- aws appsync list-data-sources --api-id [api_id]              -- testDS/None   --
 * -- cdk destroy --app 'node integ.api-import.js' stack baseStack -- clean         --
 */
const app = new cdk.App();
const baseStack = new cdk.Stack(app, 'baseStack');
const baseApi = new appsync.GraphqlApi(baseStack, 'baseApi', {
    name: 'baseApi',
    schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
});
const stack = new cdk.Stack(app, 'stack');
const api = appsync.GraphqlApi.fromGraphqlApiAttributes(stack, 'Api', {
    graphqlApiId: `${baseApi.apiId}`,
});
const testTable = new db.Table(stack, 'TestTable', {
    billingMode: db.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: db.AttributeType.STRING,
    },
    removalPolicy: cdk.RemovalPolicy.DESTROY,
});
const testDS = api.addDynamoDbDataSource('ds', testTable);
testDS.createResolver({
    typeName: 'Query',
    fieldName: 'getTests',
    requestMappingTemplate: appsync.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: appsync.MappingTemplate.dynamoDbResultList(),
});
testDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addTest',
    requestMappingTemplate: appsync.MappingTemplate.dynamoDbPutItem(appsync.PrimaryKey.partition('id').auto(), appsync.Values.projecting('test')),
    responseMappingTemplate: appsync.MappingTemplate.dynamoDbResultItem(),
});
const api2 = appsync.GraphqlApi.fromGraphqlApiAttributes(stack, 'api2', {
    graphqlApiId: baseApi.apiId,
    graphqlApiArn: baseApi.arn,
});
api2.addNoneDataSource('none');
app.synth();
//# sourceMappingURL=data:application/json;base64,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