"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const t = require("./scalar-type-defintions");
const out = 'type Test1 {\n  test1: String\n}\ntype Test2 {\n  test2: String\n}\nunion UnionTest = Test1 | Test2\n';
const test1 = new appsync.ObjectType('Test1', {
    definition: { test1: t.string },
});
const test2 = new appsync.ObjectType('Test2', {
    definition: { test2: t.string },
});
let stack;
let api;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    api = new appsync.GraphqlApi(stack, 'api', {
        name: 'api',
    });
    api.addType(test1);
    api.addType(test2);
});
describe('testing Union Type properties', () => {
    test('UnionType configures properly', () => {
        // WHEN
        const union = new appsync.UnionType('UnionTest', {
            definition: [test1, test2],
        });
        api.addType(union);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
        expect(stack).not.toHaveResource('AWS::AppSync::Resolver');
    });
    test('UnionType can addField', () => {
        // WHEN
        const union = new appsync.UnionType('UnionTest', {
            definition: [test1],
        });
        api.addType(union);
        union.addField({ field: test2.attribute() });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('UnionType errors when addField is configured with fieldName option', () => {
        // WHEN
        const union = new appsync.UnionType('UnionTest', {
            definition: [test1],
        });
        api.addType(union);
        // THEN
        expect(() => {
            union.addField({ fieldName: 'fail', field: test2.attribute() });
        }).toThrowError('Union Types cannot be configured with the fieldName option. Use the field option instead.');
    });
    test('UnionType errors when addField is not configured with field option', () => {
        // WHEN
        const union = new appsync.UnionType('UnionTest', {
            definition: [test1],
        });
        api.addType(union);
        // THEN
        expect(() => {
            union.addField({});
        }).toThrowError('Union Types must be configured with the field option.');
    });
    test('UnionType can be a GraphqlType', () => {
        // WHEN
        const union = new appsync.UnionType('UnionTest', {
            definition: [test1, test2],
        });
        api.addType(union);
        api.addType(new appsync.ObjectType('Test2', {
            definition: { union: union.attribute() },
        }));
        const obj = 'type Test2 {\n  union: UnionTest\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}${obj}`,
        });
    });
    test('appsync errors when addField with Graphql Types', () => {
        // WHEN
        const test = new appsync.UnionType('Test', {
            definition: [],
        });
        // THEN
        expect(() => {
            test.addField({ field: t.string });
        }).toThrowError('Fields for Union Types must be Object Types.');
    });
    test('appsync errors when addField with Field', () => {
        // WHEN
        const test = new appsync.UnionType('Test', {
            definition: [],
        });
        // THEN
        expect(() => {
            test.addField({ field: new appsync.Field({ returnType: t.string }) });
        }).toThrowError('Fields for Union Types must be Object Types.');
    });
    test('appsync errors when addField with ResolvableField', () => {
        // WHEN
        const test = new appsync.UnionType('Test', {
            definition: [],
        });
        // THEN
        expect(() => {
            test.addField({ field: new appsync.ResolvableField({ returnType: t.string }) });
        }).toThrowError('Fields for Union Types must be Object Types.');
    });
    test('appsync errors when addField with Interface Types', () => {
        // WHEN
        const test = new appsync.UnionType('Test', {
            definition: [],
        });
        // THEN
        expect(() => {
            test.addField({ field: new appsync.InterfaceType('break', { definition: {} }).attribute() });
        }).toThrowError('Fields for Union Types must be Object Types.');
    });
    test('appsync errors when addField with Union Types', () => {
        // WHEN
        const test = new appsync.UnionType('Test', {
            definition: [],
        });
        // THEN
        expect(() => {
            test.addField({ field: test.attribute() });
        }).toThrowError('Fields for Union Types must be Object Types.');
    });
});
//# sourceMappingURL=data:application/json;base64,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