"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const t = require("./scalar-type-defintions");
let stack;
let api;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    api = new appsync.GraphqlApi(stack, 'api', {
        name: 'api',
    });
});
describe('testing Object Type properties', () => {
    test('ObjectType can implement from interface types', () => {
        // WHEN
        const baseTest = new appsync.InterfaceType('baseTest', {
            definition: {
                id: t.id,
            },
        });
        const objectTest = new appsync.ObjectType('objectTest', {
            interfaceTypes: [baseTest],
            definition: {
                id2: t.id,
            },
            directives: [appsync.Directive.custom('@test')],
        });
        api.addType(baseTest);
        api.addType(objectTest);
        const gql_interface = 'interface baseTest {\n  id: ID\n}\n';
        const gql_object = 'type objectTest implements baseTest @test {\n  id2: ID\n  id: ID\n}\n';
        const out = `${gql_interface}${gql_object}`;
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('ObjectType can implement from multiple interface types', () => {
        // WHEN
        const baseTest = new appsync.InterfaceType('baseTest', {
            definition: { id: t.id },
        });
        const anotherTest = new appsync.InterfaceType('anotherTest', {
            definition: { id2: t.id },
        });
        const objectTest = new appsync.ObjectType('objectTest', {
            interfaceTypes: [anotherTest, baseTest],
            definition: {
                id3: t.id,
            },
        });
        api.addType(baseTest);
        api.addType(anotherTest);
        api.addType(objectTest);
        const gql_interface = 'interface baseTest {\n  id: ID\n}\ninterface anotherTest {\n  id2: ID\n}\n';
        const gql_object = 'type objectTest implements anotherTest, baseTest {\n  id3: ID\n  id2: ID\n  id: ID\n}\n';
        const out = `${gql_interface}${gql_object}`;
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can be a Graphql Type', () => {
        // WHEN
        const baseTest = new appsync.ObjectType('baseTest', {
            definition: {
                id: t.id,
            },
        });
        const graphqlType = baseTest.attribute();
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: graphqlType,
            },
        });
        api.addType(test);
        const out = 'type Test {\n  test: baseTest\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can implement Resolvable Field in definition', () => {
        // WHEN
        const field = new appsync.ResolvableField({
            returnType: t.string,
            dataSource: api.addNoneDataSource('none'),
            args: {
                arg: t.int,
            },
        });
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
                resolve: field,
            },
        });
        api.addType(test);
        const out = 'type Test {\n  test: String\n  resolve(arg: Int): String\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can implement Resolvable Field from GraphqlType', () => {
        // WHEN
        const field = new appsync.ResolvableField({
            returnType: t.string,
            dataSource: api.addNoneDataSource('none'),
            args: {
                arg: t.int,
            },
        });
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
                resolve: field,
            },
        });
        api.addType(test);
        const out = 'type Test {\n  test: String\n  resolve(arg: Int): String\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can implement Resolvable Field for pipelineResolvers', () => {
        // WHEN
        const test = new appsync.ObjectType('Test', {
            definition: {
                resolve: new appsync.ResolvableField({
                    returnType: t.string,
                    dataSource: api.addNoneDataSource('none'),
                    args: {
                        arg: t.int,
                    },
                    pipelineConfig: ['test', 'test'],
                }),
            },
        });
        api.addType(test);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::Resolver', {
            Kind: 'PIPELINE',
            PipelineConfig: { Functions: ['test', 'test'] },
        });
    });
    test('Object Type can dynamically add Fields', () => {
        // WHEN
        const field = new appsync.ResolvableField({
            returnType: t.string,
            dataSource: api.addNoneDataSource('none'),
            args: { arg: t.int },
        });
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
            },
        });
        test.addField({ fieldName: 'resolve', field });
        // test.addField('resolve', field);
        test.addField({ fieldName: 'dynamic', field: t.string });
        api.addType(test);
        const out = 'type Test {\n  test: String\n  resolve(arg: Int): String\n  dynamic: String\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
        expect(stack).toHaveResource('AWS::AppSync::Resolver');
    });
    test('Object Type can generate Fields with Directives', () => {
        // WHEN
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
            },
        });
        test.addField({
            fieldName: 'resolve',
            field: new appsync.Field({
                returnType: t.string,
                directives: [appsync.Directive.apiKey()],
            }),
        });
        api.addType(test);
        const out = 'type Test {\n  test: String\n  resolve: String\n  @aws_api_key\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can generate ResolvableFields with Directives', () => {
        // WHEN
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
            },
        });
        const field = new appsync.ResolvableField({
            returnType: t.string,
            directives: [appsync.Directive.apiKey()],
            dataSource: api.addNoneDataSource('none'),
            args: {
                arg: t.string,
            },
        });
        test.addField({ fieldName: 'resolve', field });
        api.addType(test);
        const out = 'type Test {\n  test: String\n  resolve(arg: String): String\n  @aws_api_key\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
        expect(stack).toHaveResource('AWS::AppSync::Resolver');
    });
    test('appsync fails addField with ObjectType missing fieldName', () => {
        // WHEN
        const test = new appsync.ObjectType('Test', { definition: {} });
        api.addType(test);
        // THEN
        expect(() => {
            test.addField({ fieldName: 'test' });
        }).toThrowError('Object Types must have both fieldName and field options.');
    });
    test('appsync fails addField with ObjectType missing field', () => {
        // WHEN
        const test = new appsync.ObjectType('Test', { definition: {} });
        api.addType(test);
        // THEN
        expect(() => {
            test.addField({ field: t.string });
        }).toThrowError('Object Types must have both fieldName and field options.');
    });
    test('appsync fails addField with ObjectType missing both fieldName and field options', () => {
        // WHEN
        const test = new appsync.ObjectType('Test', { definition: {} });
        api.addType(test);
        // THEN
        expect(() => {
            test.addField({});
        }).toThrowError('Object Types must have both fieldName and field options.');
    });
});
//# sourceMappingURL=data:application/json;base64,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