"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const path = require("path");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
// GLOBAL GIVEN
let stack;
let api;
beforeEach(() => {
    stack = new cdk.Stack();
    api = new appsync.GraphqlApi(stack, 'baseApi', {
        name: 'api',
        schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
    });
});
describe('None Data Source configuration', () => {
    test('default configuration produces name `NoneCDKDataSource`', () => {
        // WHEN
        api.addNoneDataSource('ds');
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'NONE',
            Name: 'ds',
        });
    });
    test('appsync configures name correctly', () => {
        // WHEN
        api.addNoneDataSource('ds', {
            name: 'custom',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'NONE',
            Name: 'custom',
        });
    });
    test('appsync configures name and description correctly', () => {
        // WHEN
        api.addNoneDataSource('ds', {
            name: 'custom',
            description: 'custom description',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'NONE',
            Name: 'custom',
            Description: 'custom description',
        });
    });
    test('appsync errors when creating multiple none data sources with no configuration', () => {
        // THEN
        expect(() => {
            api.addNoneDataSource('ds');
            api.addNoneDataSource('ds');
        }).toThrow("There is already a Construct with name 'ds' in GraphqlApi [baseApi]");
    });
    test('appsync errors when creating multiple none data sources with same name configuration', () => {
        // THEN
        expect(() => {
            api.addNoneDataSource('ds1', { name: 'custom' });
            api.addNoneDataSource('ds2', { name: 'custom' });
        }).not.toThrowError();
    });
});
describe('adding none data source from imported api', () => {
    test('imported api can add NoneDataSource from id', () => {
        // WHEN
        const importedApi = appsync.GraphqlApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
        });
        importedApi.addNoneDataSource('none');
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'NONE',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
    test('imported api can add NoneDataSource from attributes', () => {
        // WHEN
        const importedApi = appsync.GraphqlApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
            graphqlApiArn: api.arn,
        });
        importedApi.addNoneDataSource('none');
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'NONE',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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