"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const t = require("./scalar-type-defintions");
let stack;
let api;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    api = new appsync.GraphqlApi(stack, 'api', {
        name: 'api',
    });
});
describe('testing InterfaceType properties', () => {
    let baseTest;
    beforeEach(() => {
        baseTest = new appsync.InterfaceType('baseTest', {
            definition: {
                id: t.id,
            },
        });
    });
    test('basic InterfaceType produces correct schema', () => {
        // WHEN
        api.addToSchema(baseTest.toString());
        const out = 'interface baseTest {\n  id: ID\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('InterfaceType fields can have arguments', () => {
        // WHEN
        baseTest.addField({
            fieldName: 'test',
            field: new appsync.Field({
                returnType: t.string,
                args: { success: t.int },
            }),
        });
        api.addToSchema(baseTest.toString());
        const out = 'interface baseTest {\n  id: ID\n  test(success: Int): String\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('InterfaceType fields will not produce resolvers', () => {
        // WHEN
        baseTest.addField({
            fieldName: 'test',
            field: new appsync.ResolvableField({
                returnType: t.string,
                args: { success: t.int },
                dataSource: api.addNoneDataSource('none'),
            }),
        });
        api.addToSchema(baseTest.toString());
        const out = 'interface baseTest {\n  id: ID\n  test(success: Int): String\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
        expect(stack).not.toHaveResource('AWS::AppSync::Resolver');
    });
    test('Interface Type can be a Graphql Type', () => {
        // WHEN
        const graphqlType = baseTest.attribute();
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: graphqlType,
            },
        });
        api.addToSchema(test.toString());
        const out = 'type Test {\n  test: baseTest\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Interface Type can generate Fields with Directives', () => {
        // WHEN
        const test = new appsync.InterfaceType('Test', {
            definition: {
                test: t.string,
            },
        });
        test.addField({
            fieldName: 'resolve',
            field: new appsync.Field({
                returnType: t.string,
                directives: [appsync.Directive.apiKey()],
            }),
        });
        api.addType(test);
        const out = 'interface Test {\n  test: String\n  resolve: String\n  @aws_api_key\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Interface Type can generate ResolvableFields with Directives, but not the resolver', () => {
        // WHEN
        const test = new appsync.InterfaceType('Test', {
            definition: {
                test: t.string,
            },
        });
        test.addField({
            fieldName: 'resolve',
            field: new appsync.ResolvableField({
                returnType: t.string,
                directives: [appsync.Directive.apiKey()],
                dataSource: api.addNoneDataSource('none'),
            }),
        });
        api.addType(test);
        const out = 'interface Test {\n  test: String\n  resolve: String\n  @aws_api_key\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
        expect(stack).not.toHaveResource('AWS::AppSync::Resolver');
    });
    test('appsync fails addField with InterfaceType missing fieldName', () => {
        // WHEN
        const test = new appsync.InterfaceType('Test', { definition: {} });
        api.addType(test);
        // THEN
        expect(() => {
            test.addField({ fieldName: 'test' });
        }).toThrowError('Interface Types must have both fieldName and field options.');
    });
    test('appsync fails addField with InterfaceType missing field', () => {
        // WHEN
        const test = new appsync.InterfaceType('Test', { definition: {} });
        api.addType(test);
        // THEN
        expect(() => {
            test.addField({ field: t.string });
        }).toThrowError('Interface Types must have both fieldName and field options.');
    });
    test('appsync fails addField with InterfaceType missing both fieldName and field options', () => {
        // WHEN
        const test = new appsync.InterfaceType('Test', { definition: {} });
        api.addType(test);
        // THEN
        expect(() => {
            test.addField({});
        }).toThrowError('Interface Types must have both fieldName and field options.');
    });
});
//# sourceMappingURL=data:application/json;base64,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