"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const path = require("path");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
// GLOBAL GIVEN
let stack;
let api;
let endpoint;
beforeEach(() => {
    stack = new cdk.Stack();
    api = new appsync.GraphqlApi(stack, 'baseApi', {
        name: 'api',
        schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
    });
    endpoint = 'aws.amazon.com';
});
describe('Http Data Source configuration', () => {
    test('default configuration produces name `HttpCDKDataSource`', () => {
        // WHEN
        api.addHttpDataSource('ds', endpoint);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            Name: 'ds',
        });
    });
    test('appsync configures name correctly', () => {
        // WHEN
        api.addHttpDataSource('ds', endpoint, {
            name: 'custom',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            Name: 'custom',
        });
    });
    test('appsync configures name and description correctly', () => {
        // WHEN
        api.addHttpDataSource('ds', endpoint, {
            name: 'custom',
            description: 'custom description',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            Name: 'custom',
            Description: 'custom description',
        });
    });
    test('appsync configures name, authorizationConfig correctly', () => {
        // WHEN
        api.addHttpDataSource('ds', endpoint, {
            name: 'custom',
            description: 'custom description',
            authorizationConfig: {
                signingRegion: 'us-east-1',
                signingServiceName: 'states',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            Name: 'custom',
            Description: 'custom description',
            HttpConfig: {
                Endpoint: endpoint,
                AuthorizationConfig: {
                    AuthorizationType: 'AWS_IAM',
                    AwsIamConfig: {
                        SigningRegion: 'us-east-1',
                        SigningServiceName: 'states',
                    },
                },
            },
        });
    });
    test('appsync errors when creating multiple http data sources with no configuration', () => {
        // THEN
        expect(() => {
            api.addHttpDataSource('ds', endpoint);
            api.addHttpDataSource('ds', endpoint);
        }).toThrow("There is already a Construct with name 'ds' in GraphqlApi [baseApi]");
    });
});
describe('adding http data source from imported api', () => {
    test('imported api can add HttpDataSource from id', () => {
        // WHEN
        const importedApi = appsync.GraphqlApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
        });
        importedApi.addHttpDataSource('ds', endpoint);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
    test('imported api can add HttpDataSource from attributes', () => {
        // WHEN
        const importedApi = appsync.GraphqlApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
            graphqlApiArn: api.arn,
        });
        importedApi.addHttpDataSource('ds', endpoint);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'HTTP',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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