"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const cognito = require("@aws-cdk/aws-cognito");
const appsync = require("../lib");
const t = require("./scalar-type-defintions");
const iam = [appsync.Directive.iam()];
const apiKey = [appsync.Directive.apiKey()];
const oidc = [appsync.Directive.oidc()];
const cognito_default = [appsync.Directive.cognito('test', 'test2')];
const cognito_additional = [appsync.Directive.cognito('test', 'test2')];
const custom = [appsync.Directive.custom('custom')];
const generateField = (directives) => {
    return new appsync.Field({
        returnType: t.string,
        directives,
    });
};
const generateRField = (directives) => {
    return new appsync.ResolvableField({
        returnType: t.string,
        directives,
    });
};
let stack;
let api_apiKey, api_iam, api_oidc, api_auth, api_cognito;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    const userPool = new cognito.UserPool(stack, 'userpool');
    api_apiKey = new appsync.GraphqlApi(stack, 'api_apiKey', {
        name: 'api',
    });
    api_iam = new appsync.GraphqlApi(stack, 'api_iam', {
        name: 'api',
        authorizationConfig: {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.IAM,
            },
        },
    });
    api_oidc = new appsync.GraphqlApi(stack, 'api_oidc', {
        name: 'api',
        authorizationConfig: {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.OIDC,
                openIdConnectConfig: { oidcProvider: 'test' },
            },
        },
    });
    api_auth = new appsync.GraphqlApi(stack, 'api_cognito_default', {
        name: 'api',
        authorizationConfig: {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.USER_POOL,
                userPoolConfig: { userPool },
            },
        },
    });
    api_cognito = new appsync.GraphqlApi(stack, 'api_cognito_additional', {
        name: 'api',
        authorizationConfig: {
            additionalAuthorizationModes: [
                {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool },
                },
            ],
        },
    });
});
const testObjectType = (IApi, directives, tag) => {
    // WHEN
    IApi.addType(new appsync.ObjectType('Test', {
        definition: {
            field: generateField(directives),
            rfield: generateRField(directives),
        },
        directives: directives,
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
        Definition: `type Test ${tag} {\n  field: String\n  ${tag}\n  rfield: String\n  ${tag}\n}\n`,
    });
};
const testInterfaceType = (IApi, directives, tag) => {
    // WHEN
    IApi.addType(new appsync.InterfaceType('Test', {
        definition: {
            field: generateField(directives),
            rfield: generateRField(directives),
        },
        directives: directives,
    }));
    // THEN
    expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
        Definition: `interface Test ${tag} {\n  field: String\n  ${tag}\n  rfield: String\n  ${tag}\n}\n`,
    });
};
describe('Basic Testing of Directives for Code-First', () => {
    test('Iam directive configures in Object Type', () => { testObjectType(api_iam, iam, '@aws_iam'); });
    test('Iam directive configures in Interface Type', () => { testInterfaceType(api_iam, iam, '@aws_iam'); });
    test('Api Key directive configures in Object Type', () => { testObjectType(api_apiKey, apiKey, '@aws_api_key'); });
    test('Api Key directive configures in Interface Type', () => { testInterfaceType(api_apiKey, apiKey, '@aws_api_key'); });
    test('OIDC directive configures in Object Type', () => { testObjectType(api_oidc, oidc, '@aws_oidc'); });
    test('OIDC directive configures in Interface Type', () => { testInterfaceType(api_oidc, oidc, '@aws_oidc'); });
    test('Cognito as default directive configures in Object Type', () => {
        testObjectType(api_auth, cognito_default, '@aws_auth(cognito_groups: ["test", "test2"])');
    });
    test('Cognito as default directive configures in Interface Type', () => {
        testInterfaceType(api_auth, cognito_default, '@aws_auth(cognito_groups: ["test", "test2"])');
    });
    test('Cognito as additional directive configures in Object Type', () => {
        testObjectType(api_cognito, cognito_additional, '@aws_cognito_user_pools(cognito_groups: ["test", "test2"])');
    });
    test('Custom directive configures in Object Type', () => {
        testObjectType(api_cognito, custom, 'custom');
    });
    test('Custom directive configures in Interface Type', () => {
        testInterfaceType(api_cognito, custom, 'custom');
    });
});
//# sourceMappingURL=data:application/json;base64,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