"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const fs_1 = require("fs");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const private_1 = require("./private");
const schema_intermediate_1 = require("./schema-intermediate");
;
/**
 * The Schema for a GraphQL Api
 *
 * If no options are configured, schema will be generated
 * code-first.
 */
class Schema {
    constructor(options) {
        if (options === null || options === void 0 ? void 0 : options.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
        this.types = [];
    }
    /**
     * Generate a Schema from file
     *
     * @returns `SchemaAsset` with immutable schema defintion
     * @param filePath the file path of the schema file
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * Called when the GraphQL Api is initialized to allow this object to bind
     * to the stack.
     *
     * @param api The binding GraphQL Api
     */
    bind(api) {
        if (!this.schema) {
            this.schema = new appsync_generated_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: this.mode === private_1.SchemaMode.CODE ?
                    core_1.Lazy.stringValue({
                        produce: () => this.types.reduce((acc, type) => { return `${acc}${type._bindToGraphqlApi(api).toString()}\n`; }, `${this.declareSchema()}${this.definition}`),
                    })
                    : this.definition,
            });
        }
        return this.schema;
    }
    /**
     * Escape hatch to add to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add query. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.query) {
            this.query = new schema_intermediate_1.ObjectType('Query', { definition: {} });
            this.addType(this.query);
        }
        ;
        this.query.addField({ fieldName, field });
        return this.query;
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add mutation. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.mutation) {
            this.mutation = new schema_intermediate_1.ObjectType('Mutation', { definition: {} });
            this.addType(this.mutation);
        }
        ;
        this.mutation.addField({ fieldName, field });
        return this.mutation;
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        var _a, _b;
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add subscription. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.subscription) {
            this.subscription = new schema_intermediate_1.ObjectType('Subscription', { definition: {} });
            this.addType(this.subscription);
        }
        const directives = (_b = (_a = field.fieldOptions) === null || _a === void 0 ? void 0 : _a.directives) === null || _b === void 0 ? void 0 : _b.filter((directive) => directive.mutationFields);
        if (directives && directives.length > 1) {
            throw new Error(`Subscription fields must not have more than one @aws_subscribe directives. Received: ${directives.length}`);
        }
        this.subscription.addField({ fieldName, field });
        return this.subscription;
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.types.push(type);
        return type;
    }
    /**
     * Set the root types of this schema if they are defined.
     *
     * For example:
     * schema {
     *   query: Query
     *   mutation: Mutation
     *   subscription: Subscription
     * }
     */
    declareSchema() {
        if (!this.query && !this.mutation && !this.subscription) {
            return '';
        }
        const list = ['query', 'mutation', 'subscription'];
        return private_1.shapeAddition({
            prefix: 'schema',
            fields: list.map((key) => { var _a; return this[key] ? `${key}: ${(_a = this[key]) === null || _a === void 0 ? void 0 : _a.name}` : ''; })
                .filter((field) => field != ''),
        }) + '\n';
    }
}
exports.Schema = Schema;
//# sourceMappingURL=data:application/json;base64,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