import { AuthorizationType } from './graphqlapi';
import { Resolver } from './resolver';
import { Directive, IField, IIntermediateType, AddFieldOptions } from './schema-base';
import { BaseTypeOptions, GraphqlType, ResolvableFieldOptions } from './schema-field';
/**
 * Properties for configuring an Intermediate Type
 *
 * @param definition - the variables and types that define this type
 * i.e. { string: GraphqlType, string: GraphqlType }
 * @param directives - the directives for this object type
 *
 * @experimental
 */
export interface IntermediateTypeOptions {
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: IField;
    };
    /**
     * the directives for this object type
     *
     * @default - no directives
     */
    readonly directives?: Directive[];
}
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
export declare class InterfaceType implements IIntermediateType {
    /**
     * the name of this type
     */
    readonly name: string;
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: IField;
    };
    /**
     * the directives for this object type
     *
     * @default - no directives
     */
    readonly directives?: Directive[];
    /**
     * the authorization modes for this intermediate type
     */
    protected modes?: AuthorizationType[];
    constructor(name: string, props: IntermediateTypeOptions);
    /**
     * Create a GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options?: BaseTypeOptions): GraphqlType;
    /**
     * Generate the string of this object type
     */
    toString(): string;
    /**
     * Add a field to this Interface Type.
     *
     * Interface Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options: AddFieldOptions): void;
}
/**
 * Properties for configuring an Object Type
 *
 * @param definition - the variables and types that define this type
 * i.e. { string: GraphqlType, string: GraphqlType }
 * @param interfaceTypes - the interfaces that this object type implements
 * @param directives - the directives for this object type
 *
 * @experimental
 */
export interface ObjectTypeOptions extends IntermediateTypeOptions {
    /**
     * The Interface Types this Object Type implements
     *
     * @default - no interface types
     */
    readonly interfaceTypes?: InterfaceType[];
}
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
export declare class ObjectType extends InterfaceType implements IIntermediateType {
    /**
     * The Interface Types this Object Type implements
     *
     * @default - no interface types
     */
    readonly interfaceTypes?: InterfaceType[];
    /**
     * The resolvers linked to this data source
     */
    resolvers?: Resolver[];
    constructor(name: string, props: ObjectTypeOptions);
    /**
     * Add a field to this Object Type.
     *
     * Object Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options: AddFieldOptions): void;
    /**
     * Generate the string of this object type
     */
    toString(): string;
    /**
     * Generate the resolvers linked to this Object Type
     */
    protected generateResolver(fieldName: string, options?: ResolvableFieldOptions): void;
}
/**
 * Input Types are abstract types that define complex objects.
 * They are used in arguments to represent
 *
 * @experimental
 */
export declare class InputType implements IIntermediateType {
    /**
     * the name of this type
     */
    readonly name: string;
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: IField;
    };
    /**
     * the authorization modes for this intermediate type
     */
    protected modes?: AuthorizationType[];
    constructor(name: string, props: IntermediateTypeOptions);
    /**
     * Create a GraphQL Type representing this Input Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options?: BaseTypeOptions): GraphqlType;
    /**
     * Generate the string of this input type
     */
    toString(): string;
    /**
     * Add a field to this Input Type.
     *
     * Input Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options: AddFieldOptions): void;
}
/**
 * Properties for configuring an Union Type
 *
 * @experimental
 */
export interface UnionTypeOptions {
    /**
     * the object types for this union type
     */
    readonly definition: IIntermediateType[];
}
/**
 * Union Types are abstract types that are similar to Interface Types,
 * but they cannot to specify any common fields between types.
 *
 * Note that fields of a union type need to be object types. In other words,
 * you can't create a union type out of interfaces, other unions, or inputs.
 *
 * @experimental
 */
export declare class UnionType implements IIntermediateType {
    /**
     * the name of this type
     */
    readonly name: string;
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: IField;
    };
    /**
     * the authorization modes supported by this intermediate type
     */
    protected modes?: AuthorizationType[];
    constructor(name: string, options: UnionTypeOptions);
    /**
     * Create a GraphQL Type representing this Union Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options?: BaseTypeOptions): GraphqlType;
    /**
     * Generate the string of this Union type
     */
    toString(): string;
    /**
     * Add a field to this Union Type
     *
     * Input Types must have field options and the IField must be an Object Type.
     *
     * @param options the options to add a field
     */
    addField(options: AddFieldOptions): void;
}
/**
 * Properties for configuring an Enum Type
 *
 * @experimental
 */
export interface EnumTypeOptions {
    /**
     * the attributes of this type
     */
    readonly definition: string[];
}
/**
 * Enum Types are abstract types that includes a set of fields
 * that represent the strings this type can create.
 *
 * @experimental
 */
export declare class EnumType implements IIntermediateType {
    /**
     * the name of this type
     */
    readonly name: string;
    /**
     * the attributes of this type
     */
    readonly definition: {
        [key: string]: IField;
    };
    /**
     * the authorization modes for this intermediate type
     */
    protected modes?: AuthorizationType[];
    constructor(name: string, options: EnumTypeOptions);
    /**
     * Create an GraphQL Type representing this Enum Type
     */
    attribute(options?: BaseTypeOptions): GraphqlType;
    /**
     * Generate the string of this enum type
     */
    toString(): string;
    /**
     * Add a field to this Enum Type
     *
     * To add a field to this Enum Type, you must only configure
     * addField with the fieldName options.
     *
     * @param options the options to add a field
     */
    addField(options: AddFieldOptions): void;
}
