"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Between = exports.BinaryCondition = exports.BeginsWith = exports.BaseKeyCondition = exports.shapeAddition = exports.SchemaMode = void 0;
/**
 * Utility enum for Schema class
 */
var SchemaMode;
(function (SchemaMode) {
    SchemaMode["FILE"] = "FILE";
    SchemaMode["CODE"] = "CODE";
})(SchemaMode = exports.SchemaMode || (exports.SchemaMode = {}));
;
/**
 * Generates an addition to the schema
 *
 * @param options the options to produced a stringfied addition
 *
 * @returns the following shape:
 *
 * ```
 * prefix name interfaces directives {
 *   field
 *   field
 *   ...
 * }
 * ```
 */
function shapeAddition(options) {
    const typeName = () => { return options.name ? ` ${options.name}` : ''; };
    const interfaces = generateInterfaces(options.interfaceTypes);
    const directives = generateDirectives({
        directives: options.directives,
        modes: options.modes,
    });
    return options.fields.reduce((acc, field) => `${acc}  ${field}\n`, `${options.prefix}${typeName()}${interfaces}${directives} {\n`) + '}';
}
exports.shapeAddition = shapeAddition;
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
exports.BaseKeyCondition = BaseKeyCondition;
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BeginsWith = BeginsWith;
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BinaryCondition = BinaryCondition;
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
exports.Between = Between;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility function to generate interfaces for object types
 *
 * @param interfaceTypes the interfaces this object type implements
 */
function generateInterfaces(interfaceTypes) {
    if (!interfaceTypes || interfaceTypes.length === 0)
        return '';
    return interfaceTypes.reduce((acc, interfaceType) => `${acc} ${interfaceType.name},`, ' implements').slice(0, -1);
}
/**
 * Utility function to generate directives
 */
function generateDirectives(options) {
    if (!options.directives || options.directives.length === 0)
        return '';
    // reduce over all directives and get string version of the directive
    // pass in the auth modes for checks to happen on compile time
    return options.directives.reduce((acc, directive) => { var _a; return `${acc}${directive._bindToAuthModes(options.modes).toString()}${(_a = options.delimiter) !== null && _a !== void 0 ? _a : ' '}`; }, ' ').slice(0, -1);
}
//# sourceMappingURL=data:application/json;base64,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