"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            sep: '/',
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        const defaultMode = (_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) !== null && _b !== void 0 ? _b : { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = (_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.additionalAuthorizationModes) !== null && _d !== void 0 ? _d : [];
        const modes = [defaultMode, ...additionalModes];
        this.modes = modes.map((mode) => mode.authorizationType);
        this.validateAuthorizationProps(modes);
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = (_e = props.schema) !== null && _e !== void 0 ? _e : new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = (_f = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })) === null || _f === void 0 ? void 0 : _f.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _a;
        const arn = (_a = attrs.graphqlApiArn) !== null && _a !== void 0 ? _a : core_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        if (!config)
            return undefined;
        const role = new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        });
        return {
            cloudWatchLogsRoleArn: role.roleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        var _a, _b;
        if (((_a = config === null || config === void 0 ? void 0 : config.expires) === null || _a === void 0 ? void 0 : _a.isBefore(core_1.Duration.days(1))) || ((_b = config === null || config === void 0 ? void 0 : config.expires) === null || _b === void 0 ? void 0 : _b.isAfter(core_1.Duration.days(365)))) {
            throw Error('API key expiration must be between 1 and 365 days.');
        }
        const expires = (config === null || config === void 0 ? void 0 : config.expires) ? config === null || config === void 0 ? void 0 : config.expires.toEpoch() : undefined;
        return new appsync_generated_1.CfnApiKey(this, `${(config === null || config === void 0 ? void 0 : config.name) || 'Default'}ApiKey`, {
            expires,
            description: config === null || config === void 0 ? void 0 : config.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        return this.schema.addType(type);
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        return this.schema.addMutation(fieldName, field);
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        return this.schema.addSubscription(fieldName, field);
    }
}
exports.GraphqlApi = GraphqlApi;
//# sourceMappingURL=data:application/json;base64,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