# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.async_support.base.exchange import Exchange
import hashlib
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidAddress
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import NotSupported
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.errors import OnMaintenance
from ccxt.base.decimal_to_precision import TICK_SIZE
from ccxt.base.precise import Precise


class coinbasepro(Exchange):

    def describe(self):
        return self.deep_extend(super(coinbasepro, self).describe(), {
            'id': 'coinbasepro',
            'name': 'Coinbase Pro',
            'countries': ['US'],
            'rateLimit': 100,
            'userAgent': self.userAgents['chrome'],
            'pro': True,
            'has': {
                'CORS': True,
                'spot': True,
                'margin': False,
                'swap': False,
                'future': False,
                'option': False,
                'cancelAllOrders': True,
                'cancelOrder': True,
                'createDepositAddress': True,
                'createOrder': True,
                'createStopLimitOrder': True,
                'createStopMarketOrder': True,
                'createStopOrder': True,
                'fetchAccounts': True,
                'fetchBalance': True,
                'fetchClosedOrders': True,
                'fetchCurrencies': True,
                'fetchDepositAddress': None,  # the exchange does not have self method, only createDepositAddress, see https://github.com/ccxt/ccxt/pull/7405
                'fetchDeposits': True,
                'fetchLedger': True,
                'fetchMarginMode': False,
                'fetchMarkets': True,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchOrders': True,
                'fetchOrderTrades': True,
                'fetchPositionMode': False,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTime': True,
                'fetchTrades': True,
                'fetchTradingFee': False,
                'fetchTradingFees': True,
                'fetchTransactions': True,
                'fetchWithdrawals': True,
                'withdraw': True,
            },
            'timeframes': {
                '1m': 60,
                '5m': 300,
                '15m': 900,
                '1h': 3600,
                '6h': 21600,
                '1d': 86400,
            },
            'hostname': 'pro.coinbase.com',
            'urls': {
                'test': {
                    'public': 'https://api-public.sandbox.pro.coinbase.com',
                    'private': 'https://api-public.sandbox.pro.coinbase.com',
                },
                'logo': 'https://user-images.githubusercontent.com/1294454/41764625-63b7ffde-760a-11e8-996d-a6328fa9347a.jpg',
                'api': {
                    'public': 'https://api.{hostname}',
                    'private': 'https://api.{hostname}',
                },
                'www': 'https://pro.coinbase.com/',
                'doc': 'https://docs.pro.coinbase.com',
                'fees': [
                    'https://docs.pro.coinbase.com/#fees',
                    'https://support.pro.coinbase.com/customer/en/portal/articles/2945310-fees',
                ],
            },
            'requiredCredentials': {
                'apiKey': True,
                'secret': True,
                'password': True,
            },
            'api': {
                'public': {
                    'get': [
                        'currencies',
                        'products',
                        'products/{id}',
                        'products/{id}/book',
                        'products/{id}/candles',
                        'products/{id}/stats',
                        'products/{id}/ticker',
                        'products/{id}/trades',
                        'time',
                        'products/spark-lines',  # experimental
                    ],
                },
                'private': {
                    'get': [
                        'accounts',
                        'accounts/{id}',
                        'accounts/{id}/holds',
                        'accounts/{id}/ledger',
                        'accounts/{id}/transfers',
                        'coinbase-accounts',
                        'fills',
                        'funding',
                        'fees',
                        'margin/profile_information',
                        'margin/buying_power',
                        'margin/withdrawal_power',
                        'margin/withdrawal_power_all',
                        'margin/exit_plan',
                        'margin/liquidation_history',
                        'margin/position_refresh_amounts',
                        'margin/status',
                        'oracle',
                        'orders',
                        'orders/{id}',
                        'orders/client:{client_oid}',
                        'otc/orders',
                        'payment-methods',
                        'position',
                        'profiles',
                        'profiles/{id}',
                        'reports/{report_id}',
                        'transfers',
                        'transfers/{transfer_id}',
                        'users/self/exchange-limits',
                        'users/self/hold-balances',
                        'users/self/trailing-volume',
                        'withdrawals/fee-estimate',
                        'conversions/{conversion_id}',
                    ],
                    'post': [
                        'conversions',
                        'deposits/coinbase-account',
                        'deposits/payment-method',
                        'coinbase-accounts/{id}/addresses',
                        'funding/repay',
                        'orders',
                        'position/close',
                        'profiles/margin-transfer',
                        'profiles/transfer',
                        'reports',
                        'withdrawals/coinbase',
                        'withdrawals/coinbase-account',
                        'withdrawals/crypto',
                        'withdrawals/payment-method',
                    ],
                    'delete': [
                        'orders',
                        'orders/client:{client_oid}',
                        'orders/{id}',
                    ],
                },
            },
            'commonCurrencies': {
                'CGLD': 'CELO',
            },
            'precisionMode': TICK_SIZE,
            'fees': {
                'trading': {
                    'tierBased': True,  # complicated tier system per coin
                    'percentage': True,
                    'maker': 0.4 / 100,  # highest fee of all tiers
                    'taker': 0.6 / 100,  # highest fee of all tiers
                },
                'funding': {
                    'tierBased': False,
                    'percentage': False,
                    'withdraw': {
                        'BCH': 0,
                        'BTC': 0,
                        'LTC': 0,
                        'ETH': 0,
                        'EUR': 0.15,
                        'USD': 25,
                    },
                    'deposit': {
                        'BCH': 0,
                        'BTC': 0,
                        'LTC': 0,
                        'ETH': 0,
                        'EUR': 0.15,
                        'USD': 10,
                    },
                },
            },
            'exceptions': {
                'exact': {
                    'Insufficient funds': InsufficientFunds,
                    'NotFound': OrderNotFound,
                    'Invalid API Key': AuthenticationError,
                    'invalid signature': AuthenticationError,
                    'Invalid Passphrase': AuthenticationError,
                    'Invalid order id': InvalidOrder,
                    'Private rate limit exceeded': RateLimitExceeded,
                    'Trading pair not available': PermissionDenied,
                    'Product not found': InvalidOrder,
                },
                'broad': {
                    'Order already done': OrderNotFound,
                    'order not found': OrderNotFound,
                    'price too small': InvalidOrder,
                    'price too precise': InvalidOrder,
                    'under maintenance': OnMaintenance,
                    'size is too small': InvalidOrder,
                    'Cancel only mode': OnMaintenance,  # https://github.com/ccxt/ccxt/issues/7690
                },
            },
        })

    async def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = await self.publicGetCurrencies(params)
        #
        #     [
        #         {
        #             id: 'XTZ',
        #             name: 'Tezos',
        #             min_size: '0.000001',
        #             status: 'online',
        #             message: '',
        #             max_precision: '0.000001',
        #             convertible_to: [],
        #             details: {
        #                 type: 'crypto',
        #                 symbol: 'Τ',
        #                 network_confirmations: 60,
        #                 sort_order: 53,
        #                 crypto_address_link: 'https://tzstats.com/{{address}}',
        #                 crypto_transaction_link: 'https://tzstats.com/{{txId}}',
        #                 push_payment_methods: ['crypto'],
        #                 group_types: [],
        #                 display_name: '',
        #                 processing_time_seconds: 0,
        #                 min_withdrawal_amount: 1
        #             }
        #         }
        #     ]
        #
        result = {}
        for i in range(0, len(response)):
            currency = response[i]
            id = self.safe_string(currency, 'id')
            name = self.safe_string(currency, 'name')
            code = self.safe_currency_code(id)
            details = self.safe_value(currency, 'details', {})
            status = self.safe_string(currency, 'status')
            active = (status == 'online')
            result[code] = {
                'id': id,
                'code': code,
                'info': currency,
                'type': self.safe_string(details, 'type'),
                'name': name,
                'active': active,
                'deposit': None,
                'withdraw': None,
                'fee': None,
                'precision': self.safe_number(currency, 'max_precision'),
                'limits': {
                    'amount': {
                        'min': self.safe_number(details, 'min_size'),
                        'max': None,
                    },
                    'withdraw': {
                        'min': self.safe_number(details, 'min_withdrawal_amount'),
                        'max': None,
                    },
                },
            }
        return result

    async def fetch_markets(self, params={}):
        """
        retrieves data on all markets for coinbasepro
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        response = await self.publicGetProducts(params)
        #
        #     [
        #         {
        #             id: 'BTCAUCTION-USD',
        #             base_currency: 'BTC',
        #             quote_currency: 'USD',
        #             base_min_size: '0.000016',
        #             base_max_size: '1500',
        #             quote_increment: '0.01',
        #             base_increment: '0.00000001',
        #             display_name: 'BTCAUCTION/USD',
        #             min_market_funds: '1',
        #             max_market_funds: '20000000',
        #             margin_enabled: False,
        #             fx_stablecoin: False,
        #             max_slippage_percentage: '0.02000000',
        #             post_only: False,
        #             limit_only: False,
        #             cancel_only: True,
        #             trading_disabled: False,
        #             status: 'online',
        #             status_message: '',
        #             auction_mode: False
        #         },
        #         {
        #             id: 'BTC-USD',
        #             base_currency: 'BTC',
        #             quote_currency: 'USD',
        #             base_min_size: '0.000016',
        #             base_max_size: '1500',
        #             quote_increment: '0.01',
        #             base_increment: '0.00000001',
        #             display_name: 'BTC/USD',
        #             min_market_funds: '1',
        #             max_market_funds: '20000000',
        #             margin_enabled: False,
        #             fx_stablecoin: False,
        #             max_slippage_percentage: '0.02000000',
        #             post_only: False,
        #             limit_only: False,
        #             cancel_only: False,
        #             trading_disabled: False,
        #             status: 'online',
        #             status_message: '',
        #             auction_mode: False
        #         }
        #     ]
        #
        result = []
        for i in range(0, len(response)):
            market = response[i]
            id = self.safe_string(market, 'id')
            baseId, quoteId = id.split('-')
            # BTCAUCTION-USD vs BTC-USD conflict workaround, see the output sample above
            # baseId = self.safe_string(market, 'base_currency')
            # quoteId = self.safe_string(market, 'quote_currency')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            status = self.safe_string(market, 'status')
            result.append(self.extend(self.fees['trading'], {
                'id': id,
                'symbol': base + '/' + quote,
                'base': base,
                'quote': quote,
                'settle': None,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': None,
                'type': 'spot',
                'spot': True,
                'margin': self.safe_value(market, 'margin_enabled'),
                'swap': False,
                'future': False,
                'option': False,
                'active': (status == 'online'),
                'contract': False,
                'linear': None,
                'inverse': None,
                'contractSize': None,
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'amount': self.safe_number(market, 'base_increment'),
                    'price': self.safe_number(market, 'quote_increment'),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': None,
                        'max': None,
                    },
                    'price': {
                        'min': None,
                        'max': None,
                    },
                    'cost': {
                        'min': self.safe_number(market, 'min_market_funds'),
                        'max': None,
                    },
                },
                'info': market,
            }))
        return result

    async def fetch_accounts(self, params={}):
        """
        fetch all the accounts associated with a profile
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: a dictionary of `account structures <https://docs.ccxt.com/en/latest/manual.html#account-structure>` indexed by the account type
        """
        await self.load_markets()
        response = await self.privateGetAccounts(params)
        #
        #     [
        #         {
        #             id: '4aac9c60-cbda-4396-9da4-4aa71e95fba0',
        #             currency: 'BTC',
        #             balance: '0.0000000000000000',
        #             available: '0',
        #             hold: '0.0000000000000000',
        #             profile_id: 'b709263e-f42a-4c7d-949a-a95c83d065da'
        #         },
        #         {
        #             id: 'f75fa69a-1ad1-4a80-bd61-ee7faa6135a3',
        #             currency: 'USDC',
        #             balance: '0.0000000000000000',
        #             available: '0',
        #             hold: '0.0000000000000000',
        #             profile_id: 'b709263e-f42a-4c7d-949a-a95c83d065da'
        #         },
        #     ]
        #
        return self.parse_accounts(response, params)

    def parse_account(self, account):
        #
        #     {
        #         id: '4aac9c60-cbda-4396-9da4-4aa71e95fba0',
        #         currency: 'BTC',
        #         balance: '0.0000000000000000',
        #         available: '0',
        #         hold: '0.0000000000000000',
        #         profile_id: 'b709263e-f42a-4c7d-949a-a95c83d065da'
        #     }
        #
        currencyId = self.safe_string(account, 'currency')
        return {
            'id': self.safe_string(account, 'id'),
            'type': None,
            'code': self.safe_currency_code(currencyId),
            'info': account,
        }

    def parse_balance(self, response):
        result = {'info': response}
        for i in range(0, len(response)):
            balance = response[i]
            currencyId = self.safe_string(balance, 'currency')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            account['free'] = self.safe_string(balance, 'available')
            account['used'] = self.safe_string(balance, 'hold')
            account['total'] = self.safe_string(balance, 'balance')
            result[code] = account
        return self.safe_balance(result)

    async def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/en/latest/manual.html?#balance-structure>`
        """
        await self.load_markets()
        response = await self.privateGetAccounts(params)
        return self.parse_balance(response)

    async def fetch_order_book(self, symbol, limit=None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data
        :param str symbol: unified symbol of the market to fetch the order book for
        :param int|None limit: the maximum amount of order book entries to return
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/en/latest/manual.html#order-book-structure>` indexed by market symbols
        """
        await self.load_markets()
        # level 1 - only the best bid and ask
        # level 2 - top 50 bids and asks(aggregated)
        # level 3 - full order book(non aggregated)
        request = {
            'id': self.market_id(symbol),
            'level': 2,  # 1 best bidask, 2 aggregated, 3 full
        }
        response = await self.publicGetProductsIdBook(self.extend(request, params))
        #
        #     {
        #         "sequence":1924393896,
        #         "bids":[
        #             ["0.01825","24.34811287",2],
        #             ["0.01824","72.5463",3],
        #             ["0.01823","424.54298049",6],
        #         ],
        #         "asks":[
        #             ["0.01826","171.10414904",4],
        #             ["0.01827","22.60427028",1],
        #             ["0.01828","397.46018784",7],
        #         ]
        #     }
        #
        orderbook = self.parse_order_book(response, symbol)
        orderbook['nonce'] = self.safe_integer(response, 'sequence')
        return orderbook

    def parse_ticker(self, ticker, market=None):
        #
        # fetchTickers
        #
        #      [
        #         1639472400,  # timestamp
        #         4.26,  # low
        #         4.38,  # high
        #         4.35,  # open
        #         4.27  # close
        #      ]
        #
        # fetchTicker
        #
        #     publicGetProductsIdTicker
        #
        #     {
        #         "trade_id":843439,
        #         "price":"0.997999",
        #         "size":"80.29769",
        #         "time":"2020-01-28T02:13:33.012523Z",
        #         "bid":"0.997094",
        #         "ask":"0.998",
        #         "volume":"1903188.03750000"
        #     }
        #
        #     publicGetProductsIdStats
        #
        #     {
        #         "open": "34.19000000",
        #         "high": "95.70000000",
        #         "low": "7.06000000",
        #         "volume": "2.41000000"
        #     }
        #
        timestamp = None
        bid = None
        ask = None
        last = None
        high = None
        low = None
        open = None
        volume = None
        symbol = None if (market is None) else market['symbol']
        if isinstance(ticker, list):
            last = self.safe_string(ticker, 4)
            timestamp = self.milliseconds()
        else:
            timestamp = self.parse8601(self.safe_value(ticker, 'time'))
            bid = self.safe_string(ticker, 'bid')
            ask = self.safe_string(ticker, 'ask')
            high = self.safe_string(ticker, 'high')
            low = self.safe_string(ticker, 'low')
            open = self.safe_string(ticker, 'open')
            last = self.safe_string_2(ticker, 'price', 'last')
            volume = self.safe_string(ticker, 'volume')
        return self.safe_ticker({
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': high,
            'low': low,
            'bid': bid,
            'bidVolume': None,
            'ask': ask,
            'askVolume': None,
            'vwap': None,
            'open': open,
            'close': last,
            'last': last,
            'previousClose': None,
            'change': None,
            'percentage': None,
            'average': None,
            'baseVolume': volume,
            'quoteVolume': None,
            'info': ticker,
        }, market)

    async def fetch_tickers(self, symbols=None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: an array of `ticker structures <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        await self.load_markets()
        symbols = self.market_symbols(symbols)
        request = {}
        response = await self.publicGetProductsSparkLines(self.extend(request, params))
        #
        #     {
        #         YYY-USD: [
        #             [
        #                 1639472400,  # timestamp
        #                 4.26,  # low
        #                 4.38,  # high
        #                 4.35,  # open
        #                 4.27  # close
        #             ],
        #             [
        #                 1639468800,
        #                 4.31,
        #                 4.45,
        #                 4.35,
        #                 4.35
        #             ],
        #         ]
        #     }
        #
        result = {}
        marketIds = list(response.keys())
        delimiter = '-'
        for i in range(0, len(marketIds)):
            marketId = marketIds[i]
            entry = self.safe_value(response, marketId, [])
            first = self.safe_value(entry, 0, [])
            market = self.safe_market(marketId, None, delimiter)
            symbol = market['symbol']
            result[symbol] = self.parse_ticker(first, market)
        return self.filter_by_array(result, 'symbol', symbols)

    async def fetch_ticker(self, symbol, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'id': market['id'],
        }
        # publicGetProductsIdTicker or publicGetProductsIdStats
        method = self.safe_string(self.options, 'fetchTickerMethod', 'publicGetProductsIdTicker')
        response = await getattr(self, method)(self.extend(request, params))
        #
        # publicGetProductsIdTicker
        #
        #     {
        #         "trade_id":843439,
        #         "price":"0.997999",
        #         "size":"80.29769",
        #         "time":"2020-01-28T02:13:33.012523Z",
        #         "bid":"0.997094",
        #         "ask":"0.998",
        #         "volume":"1903188.03750000"
        #     }
        #
        # publicGetProductsIdStats
        #
        #     {
        #         "open": "34.19000000",
        #         "high": "95.70000000",
        #         "low": "7.06000000",
        #         "volume": "2.41000000"
        #     }
        #
        return self.parse_ticker(response, market)

    def parse_trade(self, trade, market=None):
        #
        #     {
        #         type: 'match',
        #         trade_id: 82047307,
        #         maker_order_id: '0f358725-2134-435e-be11-753912a326e0',
        #         taker_order_id: '252b7002-87a3-425c-ac73-f5b9e23f3caf',
        #         order_id: 'd50ec984-77a8-460a-b958-66f114b0de9b',
        #         side: 'sell',
        #         size: '0.00513192',
        #         price: '9314.78',
        #         product_id: 'BTC-USD',
        #         profile_id: '6244401d-c078-40d9-b305-7ad3551bc3b0',
        #         sequence: 12038915443,
        #         time: '2020-01-31T20:03:41.158814Z'
        #         created_at: '2014-11-07T22:19:28.578544Z',
        #         liquidity: 'T',
        #         fee: '0.00025',
        #         settled: True,
        #         usd_volume: '0.0924556000000000',
        #         user_id: '595eb864313c2b02ddf2937d'
        #     }
        #
        timestamp = self.parse8601(self.safe_string_2(trade, 'time', 'created_at'))
        marketId = self.safe_string(trade, 'product_id')
        market = self.safe_market(marketId, market, '-')
        feeRate = None
        takerOrMaker = None
        cost = None
        feeCurrencyId = self.safe_string_lower(market, 'quoteId')
        if feeCurrencyId is not None:
            costField = feeCurrencyId + '_value'
            cost = self.safe_string(trade, costField)
            liquidity = self.safe_string(trade, 'liquidity')
            if liquidity is not None:
                takerOrMaker = 'taker' if (liquidity == 'T') else 'maker'
                feeRate = self.safe_string(market, takerOrMaker)
        feeCost = self.safe_string_2(trade, 'fill_fees', 'fee')
        fee = {
            'cost': feeCost,
            'currency': market['quote'],
            'rate': feeRate,
        }
        id = self.safe_string(trade, 'trade_id')
        side = 'sell' if (trade['side'] == 'buy') else 'buy'
        orderId = self.safe_string(trade, 'order_id')
        # Coinbase Pro returns inverted side to fetchMyTrades vs fetchTrades
        makerOrderId = self.safe_string(trade, 'maker_order_id')
        takerOrderId = self.safe_string(trade, 'taker_order_id')
        if (orderId is not None) or ((makerOrderId is not None) and (takerOrderId is not None)):
            side = 'buy' if (trade['side'] == 'buy') else 'sell'
        price = self.safe_string(trade, 'price')
        amount = self.safe_string(trade, 'size')
        return self.safe_trade({
            'id': id,
            'order': orderId,
            'info': trade,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': market['symbol'],
            'type': None,
            'takerOrMaker': takerOrMaker,
            'side': side,
            'price': price,
            'amount': amount,
            'fee': fee,
            'cost': cost,
        }, market)

    async def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all trades made by the user
        :param str symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch trades for
        :param int|None limit: the maximum number of trades structures to retrieve
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html#trade-structure>`
        """
        # as of 2018-08-23
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchMyTrades() requires a symbol argument')
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'product_id': market['id'],
        }
        if limit is not None:
            request['limit'] = limit
        response = await self.privateGetFills(self.extend(request, params))
        return self.parse_trades(response, market, since, limit)

    async def fetch_trades(self, symbol, since=None, limit=None, params={}):
        """
        get the list of most recent trades for a particular symbol
        :param str symbol: unified symbol of the market to fetch trades for
        :param int|None since: timestamp in ms of the earliest trade to fetch
        :param int|None limit: the maximum amount of trades to fetch
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html?#public-trades>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'id': market['id'],  # fixes issue  #2
        }
        if limit is not None:
            request['limit'] = limit  # default 100
        response = await self.publicGetProductsIdTrades(self.extend(request, params))
        return self.parse_trades(response, market, since, limit)

    async def fetch_trading_fees(self, params={}):
        """
        fetch the trading fees for multiple markets
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: a dictionary of `fee structures <https://docs.ccxt.com/en/latest/manual.html#fee-structure>` indexed by market symbols
        """
        await self.load_markets()
        response = await self.privateGetFees(params)
        #
        #    {
        #        "maker_fee_rate": "0.0050",
        #        "taker_fee_rate": "0.0050",
        #        "usd_volume": "43806.92"
        #    }
        #
        maker = self.safe_number(response, 'maker_fee_rate')
        taker = self.safe_number(response, 'taker_fee_rate')
        result = {}
        for i in range(0, len(self.symbols)):
            symbol = self.symbols[i]
            result[symbol] = {
                'info': response,
                'symbol': symbol,
                'maker': maker,
                'taker': taker,
                'percentage': True,
                'tierBased': True,
            }
        return result

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     [
        #         1591514160,
        #         0.02507,
        #         0.02507,
        #         0.02507,
        #         0.02507,
        #         0.02816506
        #     ]
        #
        return [
            self.safe_timestamp(ohlcv, 0),
            self.safe_number(ohlcv, 3),
            self.safe_number(ohlcv, 2),
            self.safe_number(ohlcv, 1),
            self.safe_number(ohlcv, 4),
            self.safe_number(ohlcv, 5),
        ]

    async def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=None, params={}):
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market
        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int|None since: timestamp in ms of the earliest candle to fetch
        :param int|None limit: the maximum amount of candles to fetch
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [[int]]: A list of candles ordered as timestamp, open, high, low, close, volume
        """
        await self.load_markets()
        market = self.market(symbol)
        granularity = self.timeframes[timeframe]
        request = {
            'id': market['id'],
            'granularity': granularity,
        }
        if since is not None:
            request['start'] = self.iso8601(since)
            if limit is None:
                # https://docs.pro.coinbase.com/#get-historic-rates
                limit = 300  # max = 300
            else:
                limit = min(300, limit)
            request['end'] = self.iso8601(self.sum((limit - 1) * granularity * 1000, since))
        response = await self.publicGetProductsIdCandles(self.extend(request, params))
        #
        #     [
        #         [1591514160,0.02507,0.02507,0.02507,0.02507,0.02816506],
        #         [1591514100,0.02507,0.02507,0.02507,0.02507,1.63830323],
        #         [1591514040,0.02505,0.02507,0.02505,0.02507,0.19918178]
        #     ]
        #
        return self.parse_ohlcvs(response, market, timeframe, since, limit)

    async def fetch_time(self, params={}):
        """
        fetches the current integer timestamp in milliseconds from the exchange server
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns int: the current integer timestamp in milliseconds from the exchange server
        """
        response = await self.publicGetTime(params)
        #
        #     {
        #         "iso":"2020-05-12T08:00:51.504Z",
        #         "epoch":1589270451.504
        #     }
        #
        return self.safe_timestamp(response, 'epoch')

    def parse_order_status(self, status):
        statuses = {
            'pending': 'open',
            'active': 'open',
            'open': 'open',
            'done': 'closed',
            'canceled': 'canceled',
            'canceling': 'open',
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        #
        # createOrder
        #
        #     {
        #         "id": "d0c5340b-6d6c-49d9-b567-48c4bfca13d2",
        #         "price": "0.10000000",
        #         "size": "0.01000000",
        #         "product_id": "BTC-USD",
        #         "side": "buy",
        #         "stp": "dc",
        #         "type": "limit",
        #         "time_in_force": "GTC",
        #         "post_only": False,
        #         "created_at": "2016-12-08T20:02:28.53864Z",
        #         "fill_fees": "0.0000000000000000",
        #         "filled_size": "0.00000000",
        #         "executed_value": "0.0000000000000000",
        #         "status": "pending",
        #         "settled": False
        #     }
        #
        timestamp = self.parse8601(self.safe_string(order, 'created_at'))
        marketId = self.safe_string(order, 'product_id')
        market = self.safe_market(marketId, market, '-')
        status = self.parse_order_status(self.safe_string(order, 'status'))
        doneReason = self.safe_string(order, 'done_reason')
        if (status == 'closed') and (doneReason == 'canceled'):
            status = 'canceled'
        price = self.safe_string(order, 'price')
        filled = self.safe_string(order, 'filled_size')
        amount = self.safe_string(order, 'size', filled)
        cost = self.safe_string(order, 'executed_value')
        feeCost = self.safe_number(order, 'fill_fees')
        fee = None
        if feeCost is not None:
            fee = {
                'cost': feeCost,
                'currency': market['quote'],
                'rate': None,
            }
        id = self.safe_string(order, 'id')
        type = self.safe_string(order, 'type')
        side = self.safe_string(order, 'side')
        timeInForce = self.safe_string(order, 'time_in_force')
        postOnly = self.safe_value(order, 'post_only')
        stopPrice = self.safe_number(order, 'stop_price')
        clientOrderId = self.safe_string(order, 'client_oid')
        return self.safe_order({
            'id': id,
            'clientOrderId': clientOrderId,
            'info': order,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': None,
            'status': status,
            'symbol': market['symbol'],
            'type': type,
            'timeInForce': timeInForce,
            'postOnly': postOnly,
            'side': side,
            'price': price,
            'stopPrice': stopPrice,
            'triggerPrice': stopPrice,
            'cost': cost,
            'amount': amount,
            'filled': filled,
            'remaining': None,
            'fee': fee,
            'average': None,
            'trades': None,
        }, market)

    async def fetch_order(self, id, symbol=None, params={}):
        """
        fetches information on an order made by the user
        :param str|None symbol: not used by coinbasepro fetchOrder
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        request = {}
        clientOrderId = self.safe_string_2(params, 'clientOrderId', 'client_oid')
        method = None
        if clientOrderId is None:
            method = 'privateGetOrdersId'
            request['id'] = id
        else:
            method = 'privateGetOrdersClientClientOid'
            request['client_oid'] = clientOrderId
            params = self.omit(params, ['clientOrderId', 'client_oid'])
        response = await getattr(self, method)(self.extend(request, params))
        return self.parse_order(response)

    async def fetch_order_trades(self, id, symbol=None, since=None, limit=None, params={}):
        """
        fetch all the trades made from a single order
        :param str id: order id
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch trades for
        :param int|None limit: the maximum number of trades to retrieve
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html#trade-structure>`
        """
        await self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        request = {
            'order_id': id,
        }
        response = await self.privateGetFills(self.extend(request, params))
        return self.parse_trades(response, market, since, limit)

    async def fetch_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches information on multiple orders made by the user
        :param str|None symbol: unified market symbol of the market orders were made in
        :param int|None since: the earliest time in ms to fetch orders for
        :param int|None limit: the maximum number of  orde structures to retrieve
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        request = {
            'status': 'all',
        }
        return await self.fetch_open_orders(symbol, since, limit, self.extend(request, params))

    async def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all unfilled currently open orders
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch open orders for
        :param int|None limit: the maximum number of  open orders structures to retrieve
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        request = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['product_id'] = market['id']
        if limit is not None:
            request['limit'] = limit  # default 100
        response = await self.privateGetOrders(self.extend(request, params))
        return self.parse_orders(response, market, since, limit)

    async def fetch_closed_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches information on multiple closed orders made by the user
        :param str|None symbol: unified market symbol of the market orders were made in
        :param int|None since: the earliest time in ms to fetch orders for
        :param int|None limit: the maximum number of  orde structures to retrieve
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        request = {
            'status': 'done',
        }
        return await self.fetch_open_orders(symbol, since, limit, self.extend(request, params))

    async def create_order(self, symbol, type, side, amount, price=None, params={}):
        """
        create a trade order
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float|None price: the price at which the order is to be fullfilled, in units of the quote currency, ignored in market orders
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            # common params --------------------------------------------------
            # 'client_oid': clientOrderId,
            'type': type,
            'side': side,
            'product_id': market['id'],
            # 'size': self.amount_to_precision(symbol, amount),
            # 'stp': 'dc',  # self-trade prevention, dc = decrease and cancel, co = cancel oldest, cn = cancel newest, cb = cancel both
            # 'stop': 'loss',  # "loss" = stop loss below price, "entry" = take profit above price
            # 'stop_price': self.price_to_precision(symbol, price),
            # limit order params ---------------------------------------------
            # 'price': self.price_to_precision(symbol, price),
            # 'size': self.amount_to_precision(symbol, amount),
            # 'time_in_force': 'GTC',  # GTC, GTT, IOC, or FOK
            # 'cancel_after' [optional]* min, hour, day, requires time_in_force to be GTT
            # 'post_only': False,  # invalid when time_in_force is IOC or FOK
            # market order params --------------------------------------------
            # 'size': self.amount_to_precision(symbol, amount),
            # 'funds': self.cost_to_precision(symbol, amount),
        }
        clientOrderId = self.safe_string_2(params, 'clientOrderId', 'client_oid')
        if clientOrderId is not None:
            request['client_oid'] = clientOrderId
        stopPrice = self.safe_number_2(params, 'stopPrice', 'stop_price')
        if stopPrice is not None:
            request['stop_price'] = self.price_to_precision(symbol, stopPrice)
        timeInForce = self.safe_string_2(params, 'timeInForce', 'time_in_force')
        if timeInForce is not None:
            request['time_in_force'] = timeInForce
        postOnly = self.safe_value_2(params, 'postOnly', 'post_only', False)
        if postOnly:
            request['post_only'] = True
        params = self.omit(params, ['timeInForce', 'time_in_force', 'stopPrice', 'stop_price', 'clientOrderId', 'client_oid', 'postOnly', 'post_only'])
        if type == 'limit':
            request['price'] = self.price_to_precision(symbol, price)
            request['size'] = self.amount_to_precision(symbol, amount)
        elif type == 'market':
            cost = self.safe_number_2(params, 'cost', 'funds')
            if cost is None:
                if price is not None:
                    cost = amount * price
            else:
                params = self.omit(params, ['cost', 'funds'])
            if cost is not None:
                request['funds'] = self.cost_to_precision(symbol, cost)
            else:
                request['size'] = self.amount_to_precision(symbol, amount)
        response = await self.privatePostOrders(self.extend(request, params))
        #
        #     {
        #         "id": "d0c5340b-6d6c-49d9-b567-48c4bfca13d2",
        #         "price": "0.10000000",
        #         "size": "0.01000000",
        #         "product_id": "BTC-USD",
        #         "side": "buy",
        #         "stp": "dc",
        #         "type": "limit",
        #         "time_in_force": "GTC",
        #         "post_only": False,
        #         "created_at": "2016-12-08T20:02:28.53864Z",
        #         "fill_fees": "0.0000000000000000",
        #         "filled_size": "0.00000000",
        #         "executed_value": "0.0000000000000000",
        #         "status": "pending",
        #         "settled": False
        #     }
        #
        return self.parse_order(response, market)

    async def cancel_order(self, id, symbol=None, params={}):
        """
        cancels an open order
        :param str id: order id
        :param str|None symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        request = {
            # 'product_id': market['id'],  # the request will be more performant if you include it
        }
        clientOrderId = self.safe_string_2(params, 'clientOrderId', 'client_oid')
        method = None
        if clientOrderId is None:
            method = 'privateDeleteOrdersId'
            request['id'] = id
        else:
            method = 'privateDeleteOrdersClientClientOid'
            request['client_oid'] = clientOrderId
            params = self.omit(params, ['clientOrderId', 'client_oid'])
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['product_id'] = market['symbol']  # the request will be more performant if you include it
        return await getattr(self, method)(self.extend(request, params))

    async def cancel_all_orders(self, symbol=None, params={}):
        """
        cancel all open orders
        :param str|None symbol: unified market symbol, only orders in the market of self symbol are cancelled when symbol is not None
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        request = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['product_id'] = market['symbol']  # the request will be more performant if you include it
        return await self.privateDeleteOrders(self.extend(request, params))

    async def fetch_payment_methods(self, params={}):
        return await self.privateGetPaymentMethods(params)

    async def deposit(self, code, amount, address, params={}):
        """
        Creates a new deposit address, as required by coinbasepro
        :param str code: Unified CCXT currency code(e.g. `"USDT"`)
        :param float amount: The amount of currency to send in the deposit(e.g. `20`)
        :param str address: Not used by coinbasepro
        :param dict params: Parameters specific to the exchange API endpoint(e.g. `{"network": "TRX"}`)
        :returns: a `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        request = {
            'currency': currency['id'],
            'amount': amount,
        }
        method = 'privatePostDeposits'
        if 'payment_method_id' in params:
            # deposit from a payment_method, like a bank account
            method += 'PaymentMethod'
        elif 'coinbase_account_id' in params:
            # deposit into Coinbase Pro account from a Coinbase account
            method += 'CoinbaseAccount'
        else:
            # deposit methodotherwise we did not receive a supported deposit location
            # relevant docs link for the Googlers
            # https://docs.pro.coinbase.com/#deposits
            raise NotSupported(self.id + ' deposit() requires one of `coinbase_account_id` or `payment_method_id` extra params')
        response = await getattr(self, method)(self.extend(request, params))
        if not response:
            raise ExchangeError(self.id + ' deposit() error: ' + self.json(response))
        return {
            'info': response,
            'id': response['id'],
        }

    async def withdraw(self, code, amount, address, tag=None, params={}):
        """
        make a withdrawal
        :param str code: unified currency code
        :param float amount: the amount to withdraw
        :param str address: the address to withdraw to
        :param str|None tag:
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: a `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        tag, params = self.handle_withdraw_tag_and_params(tag, params)
        self.check_address(address)
        await self.load_markets()
        currency = self.currency(code)
        request = {
            'currency': currency['id'],
            'amount': amount,
        }
        method = 'privatePostWithdrawals'
        if 'payment_method_id' in params:
            method += 'PaymentMethod'
        elif 'coinbase_account_id' in params:
            method += 'CoinbaseAccount'
        else:
            method += 'Crypto'
            request['crypto_address'] = address
            if tag is not None:
                request['destination_tag'] = tag
        response = await getattr(self, method)(self.extend(request, params))
        if not response:
            raise ExchangeError(self.id + ' withdraw() error: ' + self.json(response))
        return self.parse_transaction(response, currency)

    def parse_ledger_entry_type(self, type):
        types = {
            'transfer': 'transfer',  # Funds moved between portfolios
            'match': 'trade',       # Funds moved as a result of a trade
            'fee': 'fee',           # Fee as a result of a trade
            'rebate': 'rebate',     # Fee rebate
            'conversion': 'trade',  # Funds converted between fiat currency and a stablecoin
        }
        return self.safe_string(types, type, type)

    def parse_ledger_entry(self, item, currency=None):
        #  {
        #      id: '12087495079',
        #      amount: '-0.0100000000000000',
        #      balance: '0.0645419900000000',
        #      created_at: '2021-10-28T17:14:32.593168Z',
        #      type: 'transfer',
        #      details: {
        #          from: '2f74edf7-1440-4586-86dc-ae58c5693691',
        #          profile_transfer_id: '3ef093ad-2482-40d1-8ede-2f89cff5099e',
        #          to: 'dda99503-4980-4b60-9549-0b770ee51336'
        #      }
        #  },
        #  {
        #     id: '11740725774',
        #     amount: '-1.7565669701255000',
        #     balance: '0.0016490047745000',
        #     created_at: '2021-10-22T03:47:34.764122Z',
        #     type: 'fee',
        #     details: {
        #         order_id: 'ad06abf4-95ab-432a-a1d8-059ef572e296',
        #         product_id: 'ETH-DAI',
        #         trade_id: '1740617'
        #     }
        #  }
        id = self.safe_string(item, 'id')
        amountString = self.safe_string(item, 'amount')
        direction = None
        afterString = self.safe_string(item, 'balance')
        beforeString = Precise.string_sub(afterString, amountString)
        if Precise.string_lt(amountString, '0'):
            direction = 'out'
            amountString = Precise.string_abs(amountString)
        else:
            direction = 'in'
        amount = self.parse_number(amountString)
        after = self.parse_number(afterString)
        before = self.parse_number(beforeString)
        timestamp = self.parse8601(self.safe_value(item, 'created_at'))
        type = self.parse_ledger_entry_type(self.safe_string(item, 'type'))
        code = self.safe_currency_code(None, currency)
        details = self.safe_value(item, 'details', {})
        account = None
        referenceAccount = None
        referenceId = None
        if type == 'transfer':
            account = self.safe_string(details, 'from')
            referenceAccount = self.safe_string(details, 'to')
            referenceId = self.safe_string(details, 'profile_transfer_id')
        else:
            referenceId = self.safe_string(details, 'order_id')
        status = 'ok'
        return {
            'id': id,
            'currency': code,
            'account': account,
            'referenceAccount': referenceAccount,
            'referenceId': referenceId,
            'status': status,
            'amount': amount,
            'before': before,
            'after': after,
            'fee': None,
            'direction': direction,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'type': type,
            'info': item,
        }

    async def fetch_ledger(self, code=None, since=None, limit=None, params={}):
        """
        fetch the history of changes, actions done by the user or operations that altered balance of the user
        :param str code: unified currency code, default is None
        :param int|None since: timestamp in ms of the earliest ledger entry, default is None
        :param int|None limit: max number of ledger entrys to return, default is None
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: a `ledger structure <https://docs.ccxt.com/en/latest/manual.html#ledger-structure>`
        """
        # https://docs.cloud.coinbase.com/exchange/reference/exchangerestapi_getaccountledger
        if code is None:
            raise ArgumentsRequired(self.id + ' fetchLedger() requires a code param')
        await self.load_markets()
        await self.load_accounts()
        currency = self.currency(code)
        accountsByCurrencyCode = self.index_by(self.accounts, 'code')
        account = self.safe_value(accountsByCurrencyCode, code)
        if account is None:
            raise ExchangeError(self.id + ' fetchLedger() could not find account id for ' + code)
        request = {
            'id': account['id'],
            # 'start_date': self.iso8601(since),
            # 'end_date': self.iso8601(self.milliseconds()),
            # 'before': 'cursor',  # sets start cursor to before date
            # 'after': 'cursor',  # sets end cursor to after date
            # 'limit': limit,  # default 100
            # 'profile_id': 'string'
        }
        if since is not None:
            request['start_date'] = self.iso8601(since)
        if limit is not None:
            request['limit'] = limit  # default 100
        response = await self.privateGetAccountsIdLedger(self.extend(request, params))
        for i in range(0, len(response)):
            response[i]['currency'] = code
        return self.parse_ledger(response, currency, since, limit)

    async def fetch_transactions(self, code=None, since=None, limit=None, params={}):
        """
        fetch history of deposits and withdrawals
        see https://docs.cloud.coinbase.com/exchange/reference/exchangerestapi_gettransfers
        see https://docs.cloud.coinbase.com/exchange/reference/exchangerestapi_getaccounttransfers
        :param str|None code: unified currency code for the currency of the transactions, default is None
        :param int|None since: timestamp in ms of the earliest transaction, default is None
        :param int|None limit: max number of transactions to return, default is None
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :param str|None params['id']: account id, when defined, the endpoint used is '/accounts/{account_id}/transfers/' instead of '/transfers/'
        :returns dict: a list of `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        await self.load_markets()
        await self.load_accounts()
        currency = None
        id = self.safe_string(params, 'id')  # account id
        if id is None:
            if code is not None:
                currency = self.currency(code)
                accountsByCurrencyCode = self.index_by(self.accounts, 'code')
                account = self.safe_value(accountsByCurrencyCode, code)
                if account is None:
                    raise ExchangeError(self.id + ' fetchTransactions() could not find account id for ' + code)
                id = account['id']
        request = {}
        if id is not None:
            request['id'] = id
        if limit is not None:
            request['limit'] = limit
        response = None
        if id is None:
            response = await self.privateGetTransfers(self.extend(request, params))
            #
            #    [
            #        {
            #            "id": "bee6fd7c-afb2-4e47-8298-671d09997d16",
            #            "type": "deposit",
            #            "created_at": "2022-12-21 00:48:45.477503+00",
            #            "completed_at": null,
            #            "account_id": "sal3802-36bd-46be-a7b8-alsjf383sldak",
            #            "user_id": "6382048209f92as392039dlks2",
            #            "amount": "0.01000000",
            #            "details": {
            #                "network": "litecoin",
            #                "crypto_address": "MKemtnCFUYKsNWaf5EMYMpwSszcXWFDtTY",
            #                "coinbase_account_id": "fl2b6925-f6ba-403n-jj03-40fl435n430f",
            #                "coinbase_transaction_id": "63a25bb13cb5cf0001d2cf17",  # withdrawals only
            #                "crypto_transaction_hash": "752f35570736341e2a253f7041a34cf1e196fc56128c900fd03d99da899d94c1",
            #                "tx_service_transaction_id": "1873249104",
            #                "coinbase_payment_method_id": ""
            #            },
            #            "canceled_at": null,
            #            "processed_at": null,
            #            "user_nonce": null,
            #            "idem": "5e3201b0-e390-5k3k-a913-c32932049242",
            #            "profile_id": "k3k302a8-c4dk-4f49-9d39-3203923wpk39",
            #            "currency": "LTC"
            #        }
            #    ]
            #
            for i in range(0, len(response)):
                account_id = self.safe_string(response[i], 'account_id')
                account = self.safe_value(self.accountsById, account_id)
                code = self.safe_string(account, 'code')
                response[i]['currency'] = code
        else:
            response = await self.privateGetAccountsIdTransfers(self.extend(request, params))
            #
            #    [
            #        {
            #            "id": "bee6fd7c-afb2-4e47-8298-671d09997d16",
            #            "type": "deposit",
            #            "created_at": "2022-12-21 00:48:45.477503+00",
            #            "completed_at": null,
            #            "amount": "0.01000000",
            #            "details": {
            #                "network": "litecoin",
            #                "crypto_address": "MKemtnCFUYKsNWaf5EMYMpwSszcXWFDtTY",
            #                "coinbase_account_id": "fl2b6925-f6ba-403n-jj03-40fl435n430f",
            #                "coinbase_transaction_id": "63a25bb13cb5cf0001d2cf17",  # withdrawals only
            #                "crypto_transaction_hash": "752f35570736341e2a253f7041a34cf1e196fc56128c900fd03d99da899d94c1",
            #                "tx_service_transaction_id": "1873249104",
            #                "coinbase_payment_method_id": ""
            #            },
            #            "canceled_at": null,
            #            "processed_at": null,
            #            "user_nonce": null,
            #            "idem": "5e3201b0-e390-5k3k-a913-c32932049242",
            #            "profile_id": "k3k302a8-c4dk-4f49-9d39-3203923wpk39",
            #            "currency": "LTC"
            #        }
            #    ]
            #
            for i in range(0, len(response)):
                response[i]['currency'] = code
        return self.parse_transactions(response, currency, since, limit)

    async def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        """
        fetch all deposits made to an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch deposits for
        :param int|None limit: the maximum number of deposits structures to retrieve
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        return await self.fetch_transactions(code, since, limit, self.extend({'type': 'deposit'}, params))

    async def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        """
        fetch all withdrawals made from an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch withdrawals for
        :param int|None limit: the maximum number of withdrawals structures to retrieve
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        return await self.fetch_transactions(code, since, limit, self.extend({'type': 'withdraw'}, params))

    def parse_transaction_status(self, transaction):
        canceled = self.safe_value(transaction, 'canceled_at')
        if canceled:
            return 'canceled'
        processed = self.safe_value(transaction, 'processed_at')
        completed = self.safe_value(transaction, 'completed_at')
        if completed:
            return 'ok'
        elif processed and not completed:
            return 'failed'
        else:
            return 'pending'

    def parse_transaction(self, transaction, currency=None):
        #
        # privateGetTransfers
        #
        #    [
        #        {
        #            "id": "bee6fd7c-afb2-4e47-8298-671d09997d16",
        #            "type": "deposit",
        #            "created_at": "2022-12-21 00:48:45.477503+00",
        #            "completed_at": null,
        #            "account_id": "sal3802-36bd-46be-a7b8-alsjf383sldak",     # only from privateGetTransfers
        #            "user_id": "6382048209f92as392039dlks2",                  # only from privateGetTransfers
        #            "amount": "0.01000000",
        #            "details": {
        #                "network": "litecoin",
        #                "crypto_address": "MKemtnCFUYKsNWaf5EMYMpwSszcXWFDtTY",
        #                "coinbase_account_id": "fl2b6925-f6ba-403n-jj03-40fl435n430f",
        #                "coinbase_transaction_id": "63a25bb13cb5cf0001d2cf17",  # withdrawals only
        #                "crypto_transaction_hash": "752f35570736341e2a253f7041a34cf1e196fc56128c900fd03d99da899d94c1",
        #                "tx_service_transaction_id": "1873249104",
        #                "coinbase_payment_method_id": ""
        #            },
        #            "canceled_at": null,
        #            "processed_at": null,
        #            "user_nonce": null,
        #            "idem": "5e3201b0-e390-5k3k-a913-c32932049242",
        #            "profile_id": "k3k302a8-c4dk-4f49-9d39-3203923wpk39",
        #            "currency": "LTC"
        #        }
        #    ]
        #
        details = self.safe_value(transaction, 'details', {})
        timestamp = self.parse8601(self.safe_string(transaction, 'created_at'))
        currencyId = self.safe_string(transaction, 'currency')
        code = self.safe_currency_code(currencyId, currency)
        amount = self.safe_number(transaction, 'amount')
        type = self.safe_string(transaction, 'type')
        address = self.safe_string(details, 'crypto_address')
        address = self.safe_string(transaction, 'crypto_address', address)
        fee = {
            'currency': None,
            'cost': None,
            'rate': None,
        }
        if type == 'withdraw':
            type = 'withdrawal'
            address = self.safe_string(details, 'sent_to_address', address)
            feeCost = self.safe_number(details, 'fee')
            if feeCost is not None:
                if amount is not None:
                    amount -= feeCost
                fee['cost'] = feeCost
                fee['currency'] = code
        networkId = self.safe_string(details, 'network')
        return {
            'info': transaction,
            'id': self.safe_string(transaction, 'id'),
            'txid': self.safe_string(details, 'crypto_transaction_hash'),
            'type': type,
            'currency': code,
            'network': self.network_id_to_code(networkId),
            'amount': amount,
            'status': self.parse_transaction_status(transaction),
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'address': address,
            'addressFrom': None,
            'addressTo': self.safe_string(details, 'crypto_address'),
            'tag': self.safe_string(details, 'destination_tag'),
            'tagFrom': None,
            'tagTo': None,
            'updated': self.parse8601(self.safe_string(transaction, 'processed_at')),
            'comment': None,
            'fee': fee,
        }

    async def create_deposit_address(self, code, params={}):
        """
        create a currency deposit address
        :param str code: unified currency code of the currency for the deposit address
        :param dict params: extra parameters specific to the coinbasepro api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/en/latest/manual.html#address-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        accounts = self.safe_value(self.options, 'coinbaseAccounts')
        if accounts is None:
            accounts = await self.privateGetCoinbaseAccounts()
            self.options['coinbaseAccounts'] = accounts  # cache it
            self.options['coinbaseAccountsByCurrencyId'] = self.index_by(accounts, 'currency')
        currencyId = currency['id']
        account = self.safe_value(self.options['coinbaseAccountsByCurrencyId'], currencyId)
        if account is None:
            # eslint-disable-next-line quotes
            raise InvalidAddress(self.id + " createDepositAddress() could not find currency code " + code + " with id = " + currencyId + " in self.options['coinbaseAccountsByCurrencyId']")
        request = {
            'id': account['id'],
        }
        response = await self.privatePostCoinbaseAccountsIdAddresses(self.extend(request, params))
        address = self.safe_string(response, 'address')
        tag = self.safe_string(response, 'destination_tag')
        return {
            'currency': code,
            'address': self.check_address(address),
            'tag': tag,
            'info': response,
        }

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        request = '/' + self.implode_params(path, params)
        query = self.omit(params, self.extract_params(path))
        if method == 'GET':
            if query:
                request += '?' + self.urlencode(query)
        url = self.implode_hostname(self.urls['api'][api]) + request
        if api == 'private':
            self.check_required_credentials()
            nonce = str(self.nonce())
            payload = ''
            if method != 'GET':
                if query:
                    body = self.json(query)
                    payload = body
            what = nonce + method + request + payload
            secret = None
            try:
                secret = self.base64_to_binary(self.secret)
            except Exception as e:
                raise AuthenticationError(self.id + ' sign() invalid base64 secret')
            signature = self.hmac(self.encode(what), secret, hashlib.sha256, 'base64')
            headers = {
                'CB-ACCESS-KEY': self.apiKey,
                'CB-ACCESS-SIGN': signature,
                'CB-ACCESS-TIMESTAMP': nonce,
                'CB-ACCESS-PASSPHRASE': self.password,
                'Content-Type': 'application/json',
            }
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, code, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if (code == 400) or (code == 404):
            if body[0] == '{':
                message = self.safe_string(response, 'message')
                feedback = self.id + ' ' + message
                self.throw_exactly_matched_exception(self.exceptions['exact'], message, feedback)
                self.throw_broadly_matched_exception(self.exceptions['broad'], message, feedback)
                raise ExchangeError(feedback)  # unknown message
            raise ExchangeError(self.id + ' ' + body)

    async def request(self, path, api='public', method='GET', params={}, headers=None, body=None, config={}, context={}):
        response = await self.fetch2(path, api, method, params, headers, body, config, context)
        if not isinstance(response, str):
            if 'message' in response:
                raise ExchangeError(self.id + ' ' + self.json(response))
        return response
