"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheInvalidator = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("./constants");
const handler_1 = require("./handler");
/**
 * Automatically invalidates paths from a CloudFront distribution based on S3
 * object updates. The invalidated paths are constructed from the S3 object keys
 * being updated, combined with an optional path prefix.
 */
class CacheInvalidator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.pathPrefix != null && !core_1.Token.isUnresolved(props.pathPrefix) && !props.pathPrefix.startsWith('/')) {
            throw new Error('The pathPrefix must start with a leading /');
        }
        const handler = new handler_1.Handler(this, 'Resource', {
            description: `Automated cache invalidation on CloudFront distribution ${props.distribution.distributionId}`,
            environment: {
                DISTRIBUTION_ID: props.distribution.distributionId,
                PATH_PREFIX: (_a = props.pathPrefix) !== null && _a !== void 0 ? _a : '/',
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(1),
        });
        handler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['cloudfront:CreateInvalidation'],
            resources: ['*'],
        }));
        handler.addEventSource(new aws_lambda_event_sources_1.S3EventSource(props.bucket, { events: [aws_s3_1.EventType.OBJECT_CREATED] }));
    }
    /**
     * The age of S3 events processed by the cache invalidator. By default this
     * is the MAXIMUM value over a 5 minutes period.
     */
    metricS3EventAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "S3EventAge" /* S3_EVENT_AGE */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.CacheInvalidator = CacheInvalidator;
//# sourceMappingURL=data:application/json;base64,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