"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const transliterator_1 = require("./transliterator");
const EFS_MOUNT_PATH = '/mnt/efs';
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const repository = props.vpcEndpoints
            ? (_a = props.codeArtifact) === null || _a === void 0 ? void 0 : _a.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
            TARGET_LANGUAGE: props.language.toString(),
            // Override $TMPDIR to be on the EFS volume (so we are not limited to 512MB)
            TMPDIR: EFS_MOUNT_PATH,
            // Override $HOME to be a fixed directory in the EFS volume (so we share npm caches)
            HOME: `${EFS_MOUNT_PATH}/HOME`,
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        const lambda = new transliterator_1.Transliterator(this, 'Resource', {
            description: `Creates ${props.language} documentation from jsii-enabled npm packages`,
            environment,
            logRetention: (_b = props.logRetention) !== null && _b !== void 0 ? _b : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        this.function = lambda;
        // TODO: The @aws-cdk/aws-lambda library does not support EFS mounts yet T_T
        lambda.node.defaultChild.addPropertyOverride('FileSystemConfigs', [{
                Arn: props.efsAccessPoint.accessPointArn,
                LocalMountPath: EFS_MOUNT_PATH,
            }]);
        props.efsAccessPoint.fileSystem.connections.allowFrom(lambda, aws_ec2_1.Port.allTraffic());
        if (props.vpcEndpoints) {
            props.vpcEndpoints.elasticFileSystem.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['elasticfilesystem:ClientMount', 'elasticfilesystem:ClientWrite'],
                conditions: {
                    Bool: { 'aws:SecureTransport': 'true' },
                    ArnEquals: { 'elasticfilesystem:AccessPointArn': props.efsAccessPoint.accessPointArn },
                },
                principals: [lambda.grantPrincipal],
                resources: [props.efsAccessPoint.fileSystem.fileSystemArn],
            }));
        }
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.function);
        // The handler reads & writes to this bucket.
        bucket.grantRead(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language, '*')}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}${constants.NOT_SUPPORTED_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
        props.monitoring.watchful.watchLambdaFunction('Transliterator Function', lambda);
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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