"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws = require("../shared/aws.lambda-shared");
const constants_1 = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
async function handler(event, _context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const bucketName = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const stateMachineArn = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    for await (const input of listIndexedPackages(bucketName)) {
        console.log(`Input payload: ${JSON.stringify(input, null, 2)}`);
        const { executionArn } = await aws.stepFunctions().startExecution({
            input: JSON.stringify(input),
            stateMachineArn,
        }).promise();
        console.log(`Started execution: ${executionArn}`);
    }
}
exports.handler = handler;
const scopedPrefix = `${constants_1.STORAGE_KEY_PREFIX}@`;
async function* listIndexedPackages(bucketName) {
    var _a;
    const request = {
        Bucket: bucketName,
        Delimiter: '/',
        Prefix: constants_1.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        request.ContinuationToken = response.NextContinuationToken;
        for (const { Prefix: prefix } of (_a = response.CommonPrefixes) !== null && _a !== void 0 ? _a : []) {
            if (prefix === null || prefix === void 0 ? void 0 : prefix.startsWith(scopedPrefix)) {
                yield* listScopedPackages(bucketName, prefix);
            }
            else if (prefix === null || prefix === void 0 ? void 0 : prefix.endsWith('/')) {
                yield* listPackageVersions(bucketName, prefix);
            }
        }
    } while (request.ContinuationToken != null);
}
async function* listScopedPackages(bucketName, prefix) {
    var _a;
    const request = {
        Bucket: bucketName,
        Delimiter: '/',
        Prefix: prefix,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        request.ContinuationToken = response.NextContinuationToken;
        for (const { Prefix: packagePrefix } of (_a = response.CommonPrefixes) !== null && _a !== void 0 ? _a : []) {
            if (packagePrefix === null || packagePrefix === void 0 ? void 0 : packagePrefix.endsWith('/')) {
                yield* listPackageVersions(bucketName, packagePrefix);
            }
        }
    } while (request.ContinuationToken != null);
}
async function* listPackageVersions(bucketName, prefix) {
    var _a;
    const request = {
        Bucket: bucketName,
        Delimiter: '/',
        Prefix: prefix,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        request.ContinuationToken = response.NextContinuationToken;
        for (let { Prefix: packageVersion } of (_a = response.CommonPrefixes) !== null && _a !== void 0 ? _a : []) {
            if (packageVersion === null || packageVersion === void 0 ? void 0 : packageVersion.endsWith('/')) {
                // Strip the trailing / so it's not duplicated in the output key.
                packageVersion = packageVersion.substring(0, packageVersion.length - 1);
                const assemblyKey = `${packageVersion}${constants_1.ASSEMBLY_KEY_SUFFIX}`;
                const metadataKey = `${packageVersion}${constants_1.METADATA_KEY_SUFFIX}`;
                const packageKey = `${packageVersion}${constants_1.PACKAGE_KEY_SUFFIX}`;
                const foundResult = await Promise.all([
                    await aws.s3ObjectExists(bucketName, assemblyKey),
                    await aws.s3ObjectExists(bucketName, metadataKey),
                    await aws.s3ObjectExists(bucketName, packageKey),
                ]);
                const allFound = foundResult.every((found) => found);
                if (allFound) {
                    const sfnInput = {
                        bucket: bucketName,
                        assembly: { key: assemblyKey },
                        metadata: { key: metadataKey },
                        package: { key: packageKey },
                    };
                    yield sfnInput;
                }
            }
        }
    } while (request.ContinuationToken != null);
}
//# sourceMappingURL=data:application/json;base64,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