"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const catalog_builder_1 = require("../catalog-builder");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const clean_up_efs_1 = require("./clean-up-efs");
const redrive_state_machine_1 = require("./redrive-state-machine");
const reprocess_all_1 = require("./reprocess-all");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT];
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], interval: core_1.Duration.seconds(30), maxAttempts: 5 })
            .addCatch(new aws_stepfunctions_1.Pass(this, 'Failed to add to catalog.json', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue));
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue.next(new aws_stepfunctions_1.Fail(this, 'Fail')))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        const efsAccessPoint = this.newEfsAccessPoint(props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => new tasks.LambdaInvoke(this, `Generate ${language} docs`, {
            lambdaFunction: new transliterator_1.Transliterator(this, `DocGen-${language}`, { ...props, efsAccessPoint, language }).function,
            outputPath: '$.result',
            resultSelector: {
                result: {
                    'language': language,
                    'success.$': '$.Payload',
                },
            },
        }).addRetry({ interval: core_1.Duration.seconds(30) })
            .addCatch(new aws_stepfunctions_1.Pass(this, `Failed ${language}`, {
            parameters: {
                'error.$': 'States.StringToJson($.Cause)',
                language,
            },
        }))))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineType: aws_stepfunctions_1.StateMachineType.STANDARD,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // Ensure the State Machine does not get to run before the VPC can be used.
        this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // This function is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the back-end.
        this.reprocessAllFunction = new reprocess_all_1.ReprocessAll(this, 'ReprocessAll', {
            description: '[ConstructHub/ReprocessAll] Reprocess all package versions through the backend',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.bucket.grantRead(this.reprocessAllFunction);
        this.stateMachine.grantStartExecution(this.reprocessAllFunction);
    }
    newEfsAccessPoint(props) {
        const fs = new aws_efs_1.FileSystem(this, 'FileSystem', {
            encrypted: true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            throughputMode: aws_efs_1.ThroughputMode.BURSTING,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        const efsAccessPoint = fs.addAccessPoint('AccessPoint', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '0777',
            },
            path: '/lambda-shared',
            posixUser: {
                uid: '1000',
                gid: '1000',
            },
        });
        efsAccessPoint.node.addDependency(fs.mountTargetsAvailable);
        const efsMountPath = '/mnt/efs';
        const cleanUp = new clean_up_efs_1.CleanUpEfs(this, 'EFSCleanUp', {
            description: '[ConstructHub/CleanUpEFS] Cleans up leftover files from an EFS file system',
            environment: {
                EFS_MOUNT_PATH: efsMountPath,
                IGNORE_DIRS: `${efsMountPath}/HOME`,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        // TODO: The @aws-cdk/aws-lambda library does not support EFS mounts yet T_T
        cleanUp.node.defaultChild.addPropertyOverride('FileSystemConfigs', [{
                Arn: efsAccessPoint.accessPointArn,
                LocalMountPath: efsMountPath,
            }]);
        fs.connections.allowFrom(cleanUp, aws_ec2_1.Port.allTraffic());
        const rule = new aws_events_1.Rule(this, 'CleanUpEFSTrigger', { description: `Runs ${cleanUp.functionName} every hour`, schedule: aws_events_1.Schedule.rate(core_1.Duration.hours(1)) });
        rule.addTarget(new aws_events_targets_1.LambdaFunction(cleanUp));
        if (props.vpcEndpoints) {
            props.vpcEndpoints.elasticFileSystem.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['elasticfilesystem:ClientMount', 'elasticfilesystem:ClientWrite'],
                conditions: {
                    Bool: { 'aws:SecureTransport': 'true' },
                    ArnEquals: { 'elasticfilesystem:AccessPointArn': efsAccessPoint.accessPointArn },
                },
                principals: [cleanUp.grantPrincipal],
                resources: [fs.fileSystemArn],
            }));
        }
        return efsAccessPoint;
    }
}
exports.Orchestration = Orchestration;
//# sourceMappingURL=data:application/json;base64,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