"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const discovery_1 = require("./discovery");
/**
 * This discovery function periodically scans the CouchDB replica of npmjs.com
 * to discover newly published packages that are relevant for indexing in the
 * Construct Hub, then notifies the ingestion function about those.
 */
class Discovery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.timeout = core_1.Duration.minutes(15);
        this.bucket = new aws_s3_1.Bucket(this, 'StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [
                {
                    prefix: "staged/" /* STAGED_KEY_PREFIX */,
                    expiration: core_1.Duration.days(30),
                },
            ],
        });
        // Note: the handler is designed to stop processing more batches about 2 minutes ahead of the timeout.
        this.function = new discovery_1.Discovery(this, 'Default', {
            description: '[ConstructHub/Discovery] Periodically query npm.js index for new construct libraries',
            environment: {
                BUCKET_NAME: this.bucket.bucketName,
                QUEUE_URL: props.queue.queueUrl,
            },
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: this.timeout,
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.bucket.grantReadWrite(this.function);
        props.queue.grantSendMessages(this.function);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(this.timeout),
            targets: [new aws_events_targets_1.LambdaFunction(this.function)],
        });
        props.monitoring.watchful.watchLambdaFunction('Discovery Function', this.function);
        this.alarmErrors = this.function.metricErrors({ period: core_1.Duration.minutes(15) }).createAlarm(this, 'ErrorsAlarm', {
            alarmDescription: 'The discovery function (on npmjs.com) failed to run',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
        this.alarmNoInvocations = this.function.metricInvocations({ period: core_1.Duration.minutes(15) })
            .createAlarm(this, 'NoInvocationsAlarm', {
            alarmDescription: 'The discovery function (on npmjs.com) is not running as scheduled',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
    /**
     * The average time it took to process a changes batch.
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of changes that were processed.
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The age of the oldest package version that was processed.
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were inspected.
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were deemed relevant.
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of time that was remaining when the lambda returned in order to
     * avoid hitting a timeout.
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of staging failures.
     */
    metricStagingFailureCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "StagingFailureCount" /* STAGING_FAILURE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The average time it took to stage a package to S3.
     */
    metricStagingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "StagingTime" /* STAGING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of changes that were not processed due to having an invalid
     * format.
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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