"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const METRICS_NAMESPACE = 'ConstructHub/CatalogBuilder';
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    console.log('Loading existing catalog...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    if (!data.Body) {
        console.log('Catalog not found. Recreating...');
        const failures = {};
        for await (const { Key: pkgKey } of relevantObjects(BUCKET_NAME)) {
            try {
                await appendPackage(packages, pkgKey, BUCKET_NAME);
            }
            catch (e) {
                failures[pkgKey] = e;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setNamespace(METRICS_NAMESPACE);
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric('FailedPackagesOnRecreation', failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    else {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
        console.log('Registering new packages...');
        // note that we intentionally don't catch errors here to let these
        // event go to the DLQ for manual inspection.
        await appendPackage(packages, event.package.key, BUCKET_NAME);
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setNamespace(METRICS_NAMESPACE);
        metrics.putMetric('RegisteredPackagesMajorVersion', catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Upload the result to S3 and exit.
    return aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: 'public, max-age=300',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: constants.STORAGE_KEY_PREFIX };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            // We only register packages if they have AT LEAST Python or TypeScript docs.
            const tsDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_TYPESCRIPT}`;
            const pyDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_PYTHON}`;
            if (!(await aws.s3ObjectExists(bucket, tsDocs)) && !(await aws.s3ObjectExists(bucket, pyDocs))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, pkgKey, bucketName) {
    var _a, _b, _c;
    console.log(`Processing key: ${pkgKey}`);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    if (found != null && version.compare(found.version) <= 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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