"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const crypto_1 = require("crypto");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const language_1 = require("./backend/shared/language");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        new aws_cloudwatch_1.Dashboard(this, 'Resource', {
            dashboardName: props.dashboardName,
            periodOverride: aws_cloudwatch_1.PeriodOverride.INHERIT,
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: '# Catalog Overview',
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                ],
                ...this.catalogOverviewLanguageSections(props.inventory),
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Discovery Function',
                            '',
                            `[button:Search Log Group](${lambdaSearchLogGroupUrl(props.discovery.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            fillMetric(props.discovery.function.metricInvocations({ label: 'Invocations' })),
                            fillMetric(props.discovery.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.discovery.metricRemainingTime({ label: 'Remaining Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(15),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Follower',
                        left: [
                            props.discovery.metricChangeCount({ label: 'Change Count' }),
                            props.discovery.metricUnprocessableEntity({ label: 'Unprocessable' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            fillMetric(props.discovery.metricNpmJsChangeAge({ label: 'Lag to npmjs.com' }), 'REPEAT'),
                            fillMetric(props.discovery.metricPackageVersionAge({ label: 'Package Version Age' }), 'REPEAT'),
                        ],
                        rightYAxis: { label: 'Milliseconds', min: 0, showUnits: false },
                        period: core_1.Duration.minutes(15),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Search Log Group](${lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                            `[button:DLQ](${sqsQueueUrl(props.ingestion.deadLetterQueue)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                            fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        leftYAxis: { label: 'Count', min: 0, showUnits: false },
                        stacked: true,
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:State Machine](${stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:DLQ](${sqsQueueUrl(props.orchestration.deadLetterQueue)})`,
                            `[button:Redrive DLQ](${lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                            `[button:Reprocess](${lambdaFunctionUrl(props.orchestration.reprocessAllFunction)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
            ],
        });
    }
    *catalogOverviewLanguageSections(inventory) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: '# Documentation Generation',
            }),
        ];
        for (const language of language_1.DocumentationLanguage.ALL) {
            yield [
                new aws_cloudwatch_1.TextWidget({
                    height: 1,
                    width: 24,
                    markdown: `## Language: ${language.toString()}`,
                }),
            ];
            yield [
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
            ];
        }
    }
}
exports.BackendDashboard = BackendDashboard;
function lambdaFunctionUrl(lambda) {
    return `/lambda/home#/functions/${lambda.functionName}`;
}
function lambdaSearchLogGroupUrl(lambda) {
    return `/cloudwatch/home#logsV2:log-groups/log-group/$252Faws$252flambda$252f${lambda.functionName}/log-events`;
}
function stateMachineUrl(stateMachine) {
    return `/states/home#/statemachines/view/${stateMachine.stateMachineArn}`;
}
function sqsQueueUrl(queue) {
    const stack = core_1.Stack.of(queue);
    // We can't use the Queue URL as-is, because we can't "easily" URL-encode it in CFN...
    return `/sqs/v2/home#/queues/https%3A%2F%2Fsqs.${stack.region}.amazonaws.com%2F${stack.account}%2F${queue.queueName}`;
}
function fillMetric(metric, value = 0) {
    // We assume namespace + name is enough to uniquely identify a metric here.
    // This is true locally at this time, but in case this ever changes, consider
    // also processing dimensions and period.
    const h = crypto_1.createHash('sha256')
        .update(metric.namespace)
        .update('\0')
        .update(metric.metricName)
        .digest('hex');
    const metricName = `m${h}`;
    return new aws_cloudwatch_1.MathExpression({
        expression: `FILL(${metricName}, ${value})`,
        label: metric.label,
        usingMetrics: { [metricName]: metric },
    });
}
//# sourceMappingURL=data:application/json;base64,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