"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const language_1 = require("../../../backend/shared/language");
const transliterator_1 = require("../../../backend/transliterator");
const repository_1 = require("../../../codeartifact/repository");
const monitoring_1 = require("../../../monitoring");
test('basic use', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    const efsFileSystem = new aws_efs_1.FileSystem(stack, 'EFS', { vpc });
    const efsAccessPoint = efsFileSystem.addAccessPoint('EFS-AP');
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        efsAccessPoint: efsAccessPoint,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
    });
    // THEN
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, vpc, efsFileSystem, efsAccessPoint),
    });
});
test('CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    const efsFileSystem = new aws_efs_1.FileSystem(stack, 'EFS', { vpc });
    const efsAccessPoint = efsFileSystem.addAccessPoint('EFS-AP');
    const codeArtifact = new repository_1.Repository(stack, 'CodeArtifact');
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact,
        efsAccessPoint,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: codeArtifact.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: codeArtifact.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: codeArtifact.repositoryNpmEndpoint,
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, codeArtifact, vpc, efsFileSystem, efsAccessPoint),
    });
});
test('VPC Endpoints', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const efsAccessPoint = new aws_efs_1.FileSystem(stack, 'EFS', { vpc, vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED } })
        .addAccessPoint('EFS-AP');
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const elasticFileSystem = vpc.addInterfaceEndpoint('EFS', {
        service: aws_ec2_1.InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM,
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        efsAccessPoint: efsAccessPoint,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, elasticFileSystem, s3 },
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, vpc),
    });
});
test('VPC Endpoints and CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const repository = new repository_1.Repository(stack, 'CodeArtifact');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
        dashboardName: 'monitoring',
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const efsFileSystem = new aws_efs_1.FileSystem(stack, 'EFS', { vpc, vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED } });
    const efsAccessPoint = efsFileSystem.addAccessPoint('EFS-AP');
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const elasticFileSystem = vpc.addInterfaceEndpoint('EFS', {
        service: aws_ec2_1.InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM,
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact: repository,
        efsAccessPoint,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, elasticFileSystem, s3 },
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: repository.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: repository.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: repository.repositoryNpmEndpoint,
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, repository, monitoring, vpc, efsFileSystem, efsAccessPoint),
    });
});
/**
 * Creates a property matcher that expects "anything" for all CfnResource
 * logicalIds found within the provided constructs.
 */
function ignoreResources(stack, ...constructs) {
    const result = {};
    for (const resource of resourcesIn(...constructs)) {
        result[stack.resolve(resource.logicalId)] = expect.anything();
    }
    return result;
    function* resourcesIn(...scopes) {
        for (const scope of scopes) {
            if (core_1.CfnResource.isCfnResource(scope)) {
                yield scope;
            }
            yield* resourcesIn(...scope.node.children);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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