"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const stream_1 = require("stream");
const zip = require("zlib");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const tar = require("tar-stream");
const catalog_builder_lambda_1 = require("../../../backend/catalog-builder/catalog-builder.lambda");
const aws = require("../../../backend/shared/aws.lambda-shared");
const constants = require("../../../backend/shared/constants");
let mockBucketName;
beforeEach((done) => {
    process.env.BUCKET_NAME = mockBucketName = crypto_1.randomBytes(16).toString('base64');
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    aws.reset();
    process.env.BUCKET_NAME = mockBucketName = undefined;
    done();
});
test('initial build', () => {
    // GIVEN
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            return cb(null, { Body: JSON.stringify(npmMetadata) });
        }
        const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    // this is the suffix that triggers the catalog builder.
    const docsSuffix = constants.DOCS_KEY_SUFFIX_TYPESCRIPT;
    const mockFirstPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${docsSuffix}` },
    ];
    const mockSecondPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${docsSuffix}` },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    AWSMock.mock('S3', 'headObject', (req, cb) => {
        const existingKeys = new Set([...mockFirstPage, ...mockSecondPage].map((obj) => obj.Key));
        if (req.Bucket === mockBucketName && existingKeys.has(req.Key)) {
            return cb(null, {});
        }
        class NotFound extends Error {
            constructor() {
                super(...arguments);
                this.code = 'NotFound';
                this.message = 'Not Found';
                this.time = new Date();
            }
        }
        return cb(new NotFound());
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('application/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '3');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '2.0.0-pre',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({
        package: {
            key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.2${constants.ASSEMBLY_KEY_SUFFIX}`,
            versionId: 'VersionID',
        },
    }, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
describe('incremental build', () => {
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    const initialScopePackageV2 = {
        description: 'Package @scope/package, version 2.3.4',
        languages: { foo: 'bar' },
        major: 2,
        metadata: npmMetadata,
        name: '@scope/package',
        version: '2.3.4',
    };
    const initialNameV1 = {
        description: 'Package name, version 1.0.0',
        languages: { foo: 'bar' },
        major: 1,
        metadata: npmMetadata,
        name: 'name',
        version: '1.0.0',
    };
    const initialNameV2 = {
        description: 'Package name, version 2.0.0-pre.10',
        languages: { foo: 'bar' },
        major: 2,
        metadata: npmMetadata,
        name: 'name',
        version: '2.0.0-pre.10',
    };
    const initialCatalog = {
        packages: [
            initialScopePackageV2,
            initialNameV1,
            initialNameV2,
        ],
        updatedAt: new Date().toISOString(),
    };
    test('new major version of @scope/package', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '4');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual([
                    initialScopePackageV2,
                    {
                        description: 'Package @scope/package, version 1.2.3',
                        languages: { foo: 'bar' },
                        major: 1,
                        metadata: npmMetadata,
                        name: '@scope/package',
                        version: '1.2.3',
                    },
                    initialNameV1,
                    initialNameV2,
                ]);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
    test('updated un-scoped package version', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '3');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual([
                    initialScopePackageV2,
                    { ...initialNameV1, description: 'Package name, version 1.2.3', version: '1.2.3' },
                    initialNameV2,
                ]);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
    test('ignored "older" minor version of @scope/package', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v2.0.5${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '3');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual(initialCatalog.packages);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
    test('ignored "older" pre-release of package', () => {
        // GIVEN
        AWSMock.mock('S3', 'getObject', (req, cb) => {
            try {
                expect(req.Bucket).toBe(mockBucketName);
            }
            catch (e) {
                return cb(e);
            }
            if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
                return cb(null, { Body: JSON.stringify(npmMetadata) });
            }
            const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
            if (matches != null) {
                mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
            }
            else if (req.Key === constants.CATALOG_KEY) {
                return cb(null, {
                    Body: JSON.stringify(initialCatalog, null, 2),
                });
            }
            else {
                return cb(new NoSuchKeyError());
            }
        });
        const event = {
            package: {
                key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre.1${constants.PACKAGE_KEY_SUFFIX}`,
            },
        };
        const mockPutObjectResult = {};
        AWSMock.mock('S3', 'putObject', (req, cb) => {
            var _a, _b;
            try {
                expect(req.Bucket).toBe(mockBucketName);
                expect(req.Key).toBe(constants.CATALOG_KEY);
                expect(req.ContentType).toBe('application/json');
                expect(req.Metadata).toHaveProperty('Package-Count', '3');
                const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
                expect(body.packages).toEqual(initialCatalog.packages);
                expect(Date.parse(body.updatedAt)).toBeDefined();
            }
            catch (e) {
                return cb(e);
            }
            return cb(null, mockPutObjectResult);
        });
        // WHEN
        const result = catalog_builder_lambda_1.handler(event, { /* context */});
        // THEN
        return expect(result).resolves.toBe(mockPutObjectResult);
    });
});
class NoSuchKeyError extends Error {
    constructor() {
        super(...arguments);
        this.code = 'NoSuchKey';
        this.time = new Date();
    }
}
function mockNpmPackage(name, version) {
    const packageJson = {
        name,
        version,
        description: `Package ${name}, version ${version}`,
        jsii: {
            targets: { foo: 'bar' },
        },
    };
    const tarball = tar.pack();
    tarball.entry({ name: 'package/ignore-me.txt' }, 'Ignore Me!');
    tarball.entry({ name: 'package/package.json' }, JSON.stringify(packageJson, null, 2));
    tarball.finalize();
    const gzip = zip.createGzip();
    tarball.pipe(gzip);
    const passthrough = new stream_1.PassThrough();
    gzip.pipe(passthrough);
    return new Promise((ok) => {
        const chunks = new Array();
        passthrough.on('data', (chunk) => chunks.push(Buffer.from(chunk)));
        passthrough.once('end', () => {
            ok(Buffer.concat(chunks));
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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