from grpc import StatusCode as GrpcStatusCode

from .helpers import parse_error_details


def to_discord_proxy_exception(ex: Exception) -> "DiscordProxyException":
    if not hasattr(ex, "details"):
        return ex
    details = parse_error_details(ex)
    status = ex.code()
    if details.type == "HTTPException":
        return DiscordProxyHttpError(
            status=details.status, code=details.code, text=details.text
        )
    elif status is GrpcStatusCode.DEADLINE_EXCEEDED:
        return DiscordProxyTimeoutError(status=status, details=ex.details())
    return DiscordProxyGrpcError(status=status, details=ex.details())


class DiscordProxyException(Exception):
    """An exception generated by Discord Proxy."""


class DiscordProxyGrpcError(DiscordProxyException):
    """gRPC error created by the gRPC protocol.

    These are usually caused by issues with the Discord Proxy gRPC server,
    e.g. a failed network connection.
    """

    def __init__(self, status: GrpcStatusCode, details: str) -> None:
        super().__init__(details)
        self._status = status
        self._details = details

    @property
    def details(self) -> str:
        """Details about the error state."""
        return self._details

    @property
    def status(self) -> "GrpcStatusCode":
        """GRPC status code. See also:
        `Status codes and their use in gRPC <https://grpc.github.io/grpc/core/md_doc_statuscodes.html>`_
        """
        return self._status

    def __str__(self) -> str:
        return (
            f"gRPC error. Status code: {self.status.name} - "
            f"Error message: {self.details}"
        )


class DiscordProxyTimeoutError(DiscordProxyGrpcError):
    """The gRPC method timed out."""


class DiscordProxyHttpError(DiscordProxyException):
    """HTTP error created by the Discord API.

    These are usually caused by API errors on the Discord server.
    """

    def __init__(self, status: int, code: int, text: str) -> None:
        super().__init__(text)
        self._status = status
        self._code = code
        self._text = text

    @property
    def code(self) -> str:
        """JSON error code."""
        return self._code

    @property
    def status(self) -> int:
        """HTTP status code."""
        return self._status

    @property
    def text(self) -> str:
        """Error message."""
        return self._text

    def __str__(self) -> str:
        return (
            f"HTTP error from the Discord API. HTTP status code: {self.status} - "
            f"JSON error code: {self.code} - Error message: {self.text}"
        )
