import ast
import builtins
import collections.abc
import importlib
import types
import typing
import warnings
from typing import *  # type: ignore

import typing_extensions

from .introspection import (
    is_parameterized_generic,
    get_generic_base_class,
    get_type_arguments,
    get_type_name,
    _get_forward_ref_code,
)
from .i_hate_circular_imports import parameterize
from ..parameter import Parameter
from ..signature_ import Signature
from ..types import Type_, TypeAnnotation, ForwardRefContext
from ..errors import *

__all__ = ["resolve_forward_refs", "annotation_to_string", "annotation_for_callable"]


@overload
def resolve_forward_refs(
    annotation: TypeAnnotation,
    module: typing.Optional[types.ModuleType] = None,
    eval_: bool = True,
    strict: bool = True,
) -> TypeAnnotation:
    ...


@overload
def resolve_forward_refs(
    annotation: TypeAnnotation,
    context: ForwardRefContext = None,
    *,
    mode: Literal["eval", "getattr", "ast"] = "eval",
    strict: bool = True,
    extra_globals: typing.Mapping[str, object] = {},
) -> TypeAnnotation:
    ...


def resolve_forward_refs(
    annotation: TypeAnnotation,
    context: ForwardRefContext = None,
    eval_: Optional[bool] = None,
    strict: bool = True,
    *,
    module: typing.Optional[types.ModuleType] = None,
    mode: Literal["eval", "getattr", "ast"] = "eval",
    extra_globals: typing.Mapping[str, object] = {},
) -> TypeAnnotation:
    """
    Resolves forward references in a type annotation.

    Examples::

        >>> resolve_forward_refs(List['int'])
        typing.List[int]
        >>> resolve_forward_refs('ellipsis')
        <class 'ellipsis'>

    Using `mode='ast'` makes partial evaluation possible::

        >>> resolve_forward_refs('List[ThisCantBeResolved'], mode='ast', strict=False)
        List['ThisCantBeResolved']

    .. versionchanged:: 1.6
        The ``module`` and ``eval_`` parameters are deprecated in favor of
        ``context`` and ``mode``.

    :param annotation: The annotation in which forward references should be resolved
    :param context: The context in which forward references will be evaluated.
        Can be a class, a function, a module, or a string representing a module
        name. If ``None``, a namespace containing some common modules like
        ``typing`` and ``collections`` will be used.
    :param mode: If ``'eval'``, references may contain arbitrary code that will
        be evaluated with ``eval``. If ``'getattr'``, they must be identifiers
        and will be resolved with ``getattr``.
    :param strict: Whether to raise an exception if a forward reference can't be resolved
    :return: A new annotation with no forward references
    """
    if module is not None:
        context = module
        warnings.warn(
            'The "module" parameter is deprecated in favor of "context"',
            DeprecationWarning,
        )

    if eval_ is not None:
        mode = "eval" if eval_ else "ast"
        warnings.warn(
            'The "eval_" parameter is deprecated in favor of "mode"',
            DeprecationWarning,
        )

    if isinstance(context, str):
        context = importlib.import_module(context)

    if isinstance(annotation, ForwardRef):
        if context is None:
            context = annotation.__forward_module__

        annotation = _get_forward_ref_code(annotation)

    if isinstance(annotation, str):
        scope = collections.ChainMap()
        scope.maps.append(extra_globals)  # type: ignore

        if context is None:
            scope.maps.extend(vars(module) for module in (builtins, typing, collections.abc, collections))  # type: ignore
        elif isinstance(context, types.ModuleType):
            scope.maps.append(vars(context))
        elif isinstance(context, str):
            module = importlib.import_module(context)
            scope.maps.append(vars(module))
        elif isinstance(context, collections.abc.Mapping):
            scope.maps.append(context)  # type: ignore
        else:
            module = importlib.import_module(context.__module__)
            scope.maps.append(vars(module))

        if mode == "eval":
            try:
                # The globals must be a real dict, so the scope will be used as
                # the locals
                annotation = eval(annotation, {}, scope)
            except Exception:
                pass
            else:
                return resolve_forward_refs(annotation, context, mode=mode, strict=strict)
        elif mode == "getattr":
            first_name, *attrs = annotation.split(".")

            try:
                value = scope[first_name]
            except KeyError:
                pass
            else:
                try:
                    for attr in attrs:
                        value = getattr(value, attr)

                    return value  # type: ignore
                except AttributeError:
                    pass
        elif mode == "ast":
            expr = ast.parse(annotation, mode="eval")

            try:
                result = _eval_ast(expr.body, scope, strict=strict)
            except _AstEvaluationFailed:
                pass
            else:
                return result  # type: ignore
        else:
            assert False, f"Invalid mode: {mode!r}"

        if annotation == "ellipsis":
            return type(...)

        if not strict:
            return annotation

        raise CannotResolveForwardref(annotation, context)

    if annotation is None:
        return None

    # In some versions, `ParamSpec` is a subclass of `list`, so make sure this
    # check happens before the `isinstance(annotation, list)` check below
    if type(annotation) in (
        typing_extensions.TypeVarTuple,
        typing_extensions.ParamSpec,
        typing_extensions.ParamSpecKwargs,
        typing_extensions.ParamSpecArgs,
    ):
        return annotation

    if isinstance(annotation, TypeVar):
        if annotation.__constraints__:
            constraints = [
                resolve_forward_refs(constraint, context, mode=mode, strict=strict)
                for constraint in annotation.__constraints__
            ]
            bound = None
        else:
            constraints = ()
            bound = resolve_forward_refs(annotation.__bound__, context, mode=mode, strict=strict)

        return TypeVar(
            annotation.__name__,  # type: ignore
            *constraints,  # type: ignore
            bound=bound,  # type: ignore
            covariant=annotation.__covariant__,  # type: ignore
            contravariant=annotation.__contravariant__,  # type: ignore
        )

    # As a special case, lists are also supported because they're used by
    # `Callable`
    if isinstance(annotation, list):
        return [
            resolve_forward_refs(typ, context, mode=mode, strict=strict) for typ in annotation
        ]  # type: ignore

    if not is_parameterized_generic(annotation, raising=False):
        return annotation

    base = get_generic_base_class(annotation)

    if base is typing_extensions.Literal:
        return annotation

    type_args = get_type_arguments(annotation)
    type_args = tuple(
        resolve_forward_refs(typ, context, mode=mode, strict=strict) for typ in type_args
    )
    return parameterize(base, type_args)


def _eval_ast(node: ast.AST, scope: typing.Mapping[str, object], strict: bool) -> object:
    # Compared to "eval" and "getattr", this method of evaluating forward refs
    # has the advantage of being able to perform partial evaluation. For
    # example, the forward ref `"ClassVar[NameThatCannotBeResolved]"` can be
    # turned into `ClassVar["NameThatCannotBeResolved"]`.
    #
    # Sometimes we need to know whether the forward ref was resolved or not.
    # That's why this function returns a tuple of `(bool, object)`.
    def recurse(node: ast.AST) -> object:
        return _eval_ast(node, scope, strict)

    if strict:
        safe_recurse = recurse
    else:

        def safe_recurse(node: ast.AST) -> object:
            try:
                return _eval_ast(node, scope, strict)
            except Exception:
                return ast.unparse(node)

    if type(node) is ast.Name:
        name = node.id
        return scope[name]
    elif type(node) is ast.Attribute:
        obj = recurse(node.value)
        return getattr(obj, node.attr)
    elif type(node) is ast.Subscript:
        generic_type = recurse(node.value)
        subtype = safe_recurse(node.slice)
        return generic_type[subtype]  # type: ignore
    elif type(node) is ast.Constant:
        return node.value
    elif type(node) is ast.Tuple:
        return tuple(safe_recurse(elt) for elt in node.elts)
    elif type(node) is ast.List:
        return [safe_recurse(elt) for elt in node.elts]
    elif type(node) is ast.BinOp:
        if type(node.op) is ast.BitOr:
            left = safe_recurse(node.left)
            right = safe_recurse(node.right)
            # Use `Union` instead of `|` because:
            # 1. It works in older python versions
            # 2. It works even if `left` and `right` are strings because they
            #    couldn't be resolved
            return Union[left, right]  # type: ignore

    raise NotImplementedError(f"Can't evaluate AST node {node}")


def annotation_to_string(
    annotation: TypeAnnotation,
    *,
    implicit_typing: bool = True,
    new_style_unions: bool = True,
    optional_as_union: bool = True,
    variance_prefixes: bool = False,
) -> str:
    """
    Converts a type annotation to string. The result is valid python code
    (unless ``variance_prefixes`` is set to ``True``).

    Examples::

        >>> annotation_to_string(int)
        'int'
        >>> annotation_to_string(None)
        'None'
        >>> annotation_to_string(typing.List[int])
        'List[int]'

    :param annotation: A class or type annotation
    :param implicit_typing: Whether to omit the "typing." prefix from ``typing``
        types' names
    :param new_style_unions: Whether to use the new-style ``typing.Union`` syntax
        ``int | str`` instead of ``Union[int, str]``
    :param variance_prefixes: Whether `TypeVars` and `ParamSpecs` should be
        prefixed with ``+``, ``-``, or ``~`` to indicate variance
    :return: A string that, when evaluated, returns ``annotation``
    """

    def recurse(annotation: TypeAnnotation) -> str:
        return annotation_to_string(
            annotation,
            implicit_typing=implicit_typing,
            new_style_unions=new_style_unions,
            optional_as_union=optional_as_union,
            variance_prefixes=variance_prefixes,
        )

    def process_nested(prefix: str, elems: Iterable[TypeAnnotation]):
        elems = ", ".join(recurse(ann) for ann in elems)
        return "{}[{}]".format(prefix, elems)

    if isinstance(annotation, list):
        return process_nested("", annotation)

    if isinstance(annotation, ForwardRef):
        return repr(_get_forward_ref_code(annotation))

    if annotation is ...:
        return "..."

    if annotation is type(None):
        return "None"

    if is_parameterized_generic(annotation, raising=False):
        base = get_generic_base_class(annotation)
        subtypes = get_type_arguments(annotation)

        if base is typing.Optional and optional_as_union:
            base = typing.Union
            subtypes = [subtypes[0], None]

        if base is typing.Union and new_style_unions:
            return " | ".join(recurse(sub) for sub in subtypes)

        prefix = recurse(base)
        return process_nested(prefix, subtypes)

    if isinstance(annotation, (typing.TypeVar, typing_extensions.ParamSpec)):
        result = annotation.__name__

        if variance_prefixes:
            if annotation.__covariant__:
                result = "+" + result
            elif annotation.__contravariant__:
                result = "-" + result
            else:
                result = "~" + result

        return result

    if isinstance(annotation, typing_extensions.TypeVarTuple):
        return annotation.__name__

    if isinstance(annotation, typing_extensions.ParamSpecArgs):
        return recurse(annotation.__origin__) + ".args"

    if isinstance(annotation, typing_extensions.ParamSpecKwargs):
        return recurse(annotation.__origin__) + ".kwargs"

    if hasattr(annotation, "__module__"):
        if annotation.__module__ == "builtins":
            return annotation.__qualname__
        elif annotation.__module__ in ("typing", "typing_extensions"):
            annotation = get_type_name(annotation)

            if not implicit_typing:
                annotation = "typing." + annotation

            return annotation
        else:
            return "{}.{}".format(annotation.__module__, annotation.__qualname__)

    return repr(annotation)


def annotation_for_callable(callable_: typing.Callable[..., object]) -> Type_:
    """
    Given a callable object as input, returns a matching type annotation.

    Examples::

        >>> annotation_for_callable(len)
        typing.Callable[[typing.Sized], int]

    Note: How ``*args``, ``**kwargs``, and keyword-only parameters are handled
    is currently undefined.

    .. versionadded:: 1.5
    """
    signature = Signature.from_callable(callable_)
    parameters = signature.parameters.values()

    if signature.return_annotation is Signature.empty:
        return_type = typing.Any
    else:
        return_type = signature.return_annotation

    param_types = [
        typing.Any if param.annotation is Parameter.empty else param.annotation
        for param in parameters
        if param.kind <= Parameter.POSITIONAL_OR_KEYWORD
    ]  # TODO: Raise an exception if keyword-only parameters exist?

    # If some parameters are optional, we have to return a Union of Callable
    # types with fewer and fewer parameters
    options = [param_types]

    for param in reversed(parameters):
        if not param.is_optional:
            break

        param_types = param_types[:-1]
        options.append(param_types)

    if len(options) == 1:
        return typing.Callable[param_types, return_type]  # type: ignore

    options = tuple(typing.Callable[option, return_type] for option in options)  # type: ignore
    return typing.Union[options]  # type: ignore
