# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['flask_alchemydumps']

package_data = \
{'': ['*']}

install_requires = \
['Flask-SQLAlchemy>=2.4.1,<3.0.0',
 'Flask>=1.1.2,<2.0.0',
 'SQLAlchemy>=1.3.16,<2.0.0',
 'python-decouple>=3.3,<4.0']

setup_kwargs = {
    'name': 'flask-alchemydumps',
    'version': '0.0.12',
    'description': 'SQLAlchemy backup/dump tool for Flask',
    'long_description': '# AlchemyDumps [![Latest release](https://img.shields.io/pypi/v/Flask-AlchemyDumps.svg?style=flat)](https://pypi.python.org/pypi/Flask-AlchemyDumps)\n\n[![Development Status: Alpha](https://img.shields.io/pypi/status/Flask-AlchemyDumps.svg?style=flat)](https://pypi.python.org/pypi/Flask-AlchemyDumps)\n[![Python Version](https://img.shields.io/pypi/pyversions/Flask-AlchemyDumps.svg)](https://pypi.python.org/pypi/Flask-AlchemyDumps)\n[![Travis CI](https://img.shields.io/travis/cuducos/alchemydumps.svg?style=flat)](https://travis-ci.org/cuducos/alchemydumps)\n[![Coverage Status](https://coveralls.io/repos/github/cuducos/alchemydumps/badge.svg?branch=master)](https://coveralls.io/github/cuducos/alchemydumps?branch=master)\n\n\nDo you use [Flask](http://flask.pocoo.org>) with [SQLAlchemy](http://www.sqlalchemy.org/)? Wow, what a coincidence!\n\nThis package lets you backup and restore all your data using [SQLAlchemy dumps() method](http://docs.sqlalchemy.org/en/latest/core/serializer.html).\n\nIt is an easy way (one single command, I mean it) to save **all** the data stored in your database.\n\nYou can save it locally or in a remote server via FTP.\n\n> **WARNING** [**@baumatron**](https://github.com/baumatron)\'ve [found an important bug](https://github.com/cuducos/alchemydumps/issues/13): at this time this package won\'t backup [non-standard mappings](http://docs.sqlalchemy.org/en/latest/orm/nonstandard_mappings.html), such as [many-to-many association tables](http://docs.sqlalchemy.org/en/latest/orm/basic_relationships.html#many-to-many). This is still an open issue and I have no expectation to fix is in the following weeks — **_pull requests_ are more tham welcomed**.\n\n## Install\n\nFirst install the package: `pip install Flask-AlchemyDumps`\n\nThen pass your Flask application and SQLALchemy database to it.\n\nFor example:\n\n* The ***second line*** imports the object from the package.\n* The ***last line*** instantiates `AlchemyDumps` for your app and database.\n\n```python\nfrom flask import Flask\nfrom flask_alchemydumps import AlchemyDumps\nfrom flask_sqlalchemy import SQLAlchemy\n\n# init Flask\napp = Flask(__name__)\n\n# init SQLAlchemy and Flask-Script\napp.config[\'SQLALCHEMY_DATABASE_URI\'] = \'sqlite:///app.db\'\ndb = SQLAlchemy(app)\n\n# init Alchemy Dumps\nalchemydumps = AlchemyDumps(app, db)\n```\n\n### Remote backup (via FTP)\n\nIf you want to save your backup in a remote server via FTP, just make sure to set these environment variables replacing the placeholder information with the proper credentials:\n\n```console\nexport ALCHEMYDUMPS_FTP_SERVER=ftp.server.com\nexport ALCHEMYDUMPS_FTP_USER=johndoe\nexport ALCHEMYDUMPS_FTP_PASSWORD=secret\nexport ALCHEMYDUMPS_FTP_PATH=/absolute/path/\n```\n\nIf you want, there is a `.env.sample` inside the `/tests` folder. Just copy it to your application root folder, rename it to `.env`, and insert your credentials.\n\n### Using application factory\n\nIt is possible to use this package with application factories:\n\n```python\nalchemydumps = AlchemyDumps()\n\ndef create_app(settings):\n    …\n    alchemydumps.init_app(app, db)\n    …\n```\n\n### .gitignore\n\nYou might want to add `alchemydumps/` to your `.gitignore`. It is the folder where **AlchemyDumps** save the backup files.\n\n## Examples\n\nConsidering you have these *models* — that is to say, these [SQLAlchemy mapped classes](http://docs.sqlalchemy.org/en/latest/orm/mapper_config.html):\n\n```python\nclass User(db.Model):\n    id = db.Column(db.Integer, primary_key=True)\n    email = db.Column(db.String(140), index=True, unique=True)\n    posts = db.relationship(\'Post\', backref=\'author\', lazy=\'dynamic\')\n\nclass Post(db.Model):\n    id = db.Column(db.Integer, primary_key=True)\n    title = db.Column(db.String(140))\n    content = db.Column(db.UnicodeText)\n    author_id = db.Column(db.Integer, db.ForeignKey(\'user.id\'))\n```\n\nJust in case: this is a simple example, but you can use *abstract* mapped classes as well.\n\n### You can backup all your data\n\n```console\npython manage.py alchemydumps create\n```\n\nOutput:\n\n```\n==> 3 rows from User saved as /vagrant/alchemydumps/db-bkp-20141115172107-User.gz\n==> 42 rows from Post saved as /vagrant/alchemydumps/db-bkp-20141115172107-Post.gz\n```\n\n### You can list the backups you have already created\n\n```console\npython manage.py alchemydumps history\n```\n\nOutput:\n\t\n```\n==> ID: 20141114203949 (from Nov 15, 2014 at 17:21:07)\n    /vagrant/alchemydumps/db-bkp-20141115172107-User.gz\n    /vagrant/alchemydumps/db-bkp-20141115172107-Post.gz\n\n==> ID: 20141115140629 (from Nov 15, 2014 at 14:06:29)\n    /vagrant/alchemydumps/db-bkp-20141115140629-User.gz\n    /vagrant/alchemydumps/db-bkp-20141115140629-Post.gz\n```\n\n### You can restore a backup\n\n```console\npython manage.py alchemydumps restore -d 20141115172107\n```\n\nOutput:\n\n```\n==> db-bkp-20141115172107-User.gz totally restored.\n==> db-bkp-20141115172107-Post.gz totally restored.\n```\n\n### You can delete an existing backup\n\n```console\npython manage.py alchemydumps remove -d 20141115172107\n```\n\nOutput:\n\n```\n==> Do you want to delete the following files?\n    /vagrant/alchemydumps/db-bkp-20141115172107-User.gz\n    /vagrant/alchemydumps/db-bkp-20141115172107-Post.gz\n==> Press "Y" to confirm, or anything else to abort: y\n    db-bkp-20141115172107-User.gz deleted.\n    db-bkp-20141115172107-Post.gz deleted.\n```\n\n### And you can use the auto-clean command\n\nThe `autoclean` command follows these rules to delete backups:\n\n* It keeps **all** the backups from the last 7 days.\n* It keeps **the most recent** backup **from each week of the last month**.\n* It keeps **the most recent** backup **from each month of the last year**.\n* It keeps **the most recent** backup **from each remaining year**.\n\n```console\npython manage.py alchemydumps autoclean\n```\n\nOutput:\n\n```\n==> 8 backups will be kept:\n\n    ID: 20130703225903 (from Jul 03, 2013 at 22:59:03)\n    /vagrant/alchemydumps/db-bkp-20130703225903-User.gz\n    /vagrant/alchemydumps/db-bkp-20130703225903-Post.gz\n\n    ID: 20120405013054 (from Apr 05, 2012 at 01:30:54)\n    /vagrant/alchemydumps/db-bkp-20120405013054-User.gz\n    /vagrant/alchemydumps/db-bkp-20120405013054-Post.gz\n\n    ID: 20101123054342 (from Nov 23, 2010 at 05:43:42)\n    /vagrant/alchemydumps/db-bkp-20101123054342-User.gz\n    /vagrant/alchemydumps/db-bkp-20101123054342-Post.gz\n\n    ID: 20090708100815 (from Jul 08, 2009 at 10:08:15)\n    /vagrant/alchemydumps/db-bkp-20090708100815-User.gz\n    /vagrant/alchemydumps/db-bkp-20090708100815-Post.gz\n\n    ID: 20081208191908 (from Dec 08, 2008 at 19:19:08)\n    /vagrant/alchemydumps/db-bkp-20081208191908-User.gz\n    /vagrant/alchemydumps/db-bkp-20081208191908-Post.gz\n\n    ID: 20070114122922 (from Jan 14, 2007 at 12:29:22)\n    /vagrant/alchemydumps/db-bkp-20070114122922-User.gz\n    /vagrant/alchemydumps/db-bkp-20070114122922-Post.gz\n\n    ID: 20060911035318 (from Sep 11, 2006 at 03:53:18)\n    /vagrant/alchemydumps/db-bkp-20060911035318-User.gz\n    /vagrant/alchemydumps/db-bkp-20060911035318-Post.gz\n\n    ID: 20051108082503 (from Nov 08, 2005 at 08:25:03)\n    /vagrant/alchemydumps/db-bkp-20051108082503-User.gz\n    /vagrant/alchemydumps/db-bkp-20051108082503-Post.gz\n\n==> 11 backups will be deleted:\n\n    ID: 20120123032442 (from Jan 23, 2012 at 03:24:42)\n    /vagrant/alchemydumps/db-bkp-20120123032442-User.gz\n    /vagrant/alchemydumps/db-bkp-20120123032442-Post.gz\n\n    ID: 20101029100412 (from Oct 29, 2010 at 10:04:12)\n    /vagrant/alchemydumps/db-bkp-20101029100412-User.gz\n    /vagrant/alchemydumps/db-bkp-20101029100412-Post.gz\n\n    ID: 20100526185156 (from May 26, 2010 at 18:51:56)\n    /vagrant/alchemydumps/db-bkp-20100526185156-User.gz\n    /vagrant/alchemydumps/db-bkp-20100526185156-Post.gz\n\n    ID: 20100423085529 (from Apr 23, 2010 at 08:55:29)\n    /vagrant/alchemydumps/db-bkp-20100423085529-User.gz\n    /vagrant/alchemydumps/db-bkp-20100423085529-Post.gz\n\n    ID: 20081006074458 (from Oct 06, 2008 at 07:44:58)\n    /vagrant/alchemydumps/db-bkp-20081006074458-User.gz\n    /vagrant/alchemydumps/db-bkp-20081006074458-Post.gz\n\n    ID: 20080429210254 (from Apr 29, 2008 at 21:02:54)\n    /vagrant/alchemydumps/db-bkp-20080429210254-User.gz\n    /vagrant/alchemydumps/db-bkp-20080429210254-Post.gz\n\n    ID: 20080424043716 (from Apr 24, 2008 at 04:37:16)\n    /vagrant/alchemydumps/db-bkp-20080424043716-User.gz\n    /vagrant/alchemydumps/db-bkp-20080424043716-Post.gz\n\n    ID: 20080405110244 (from Apr 05, 2008 at 11:02:44)\n    /vagrant/alchemydumps/db-bkp-20080405110244-User.gz\n    /vagrant/alchemydumps/db-bkp-20080405110244-Post.gz\n\n    ID: 20060629054914 (from Jun 29, 2006 at 05:49:14)\n    /vagrant/alchemydumps/db-bkp-20060629054914-User.gz\n    /vagrant/alchemydumps/db-bkp-20060629054914-Post.gz\n\n    ID: 20060329020048 (from Mar 29, 2006 at 02:00:48)\n    /vagrant/alchemydumps/db-bkp-20060329020048-User.gz\n    /vagrant/alchemydumps/db-bkp-20060329020048-Post.gz\n\n    ID: 20050324012859 (from Mar 24, 2005 at 01:28:59)\n    /vagrant/alchemydumps/db-bkp-20050324012859-User.gz\n    /vagrant/alchemydumps/db-bkp-20050324012859-Post.gz\n\n==> Press "Y" to confirm, or anything else to abort. y\n    db-bkp-20120123032442-User.gz deleted.\n    db-bkp-20120123032442-Post.gz deleted.\n    db-bkp-20101029100412-User.gz deleted.\n    db-bkp-20101029100412-Post.gz deleted.\n    db-bkp-20100526185156-User.gz deleted.\n    db-bkp-20100526185156-Post.gz deleted.\n    db-bkp-20100423085529-User.gz deleted.\n    db-bkp-20100423085529-Post.gz deleted.\n    db-bkp-20081006074458-User.gz deleted.\n    db-bkp-20081006074458-Post.gz deleted.\n    db-bkp-20080429210254-User.gz deleted.\n    db-bkp-20080429210254-Post.gz deleted.\n    db-bkp-20080424043716-User.gz deleted.\n    db-bkp-20080424043716-Post.gz deleted.\n    db-bkp-20080405110244-User.gz deleted.\n    db-bkp-20080405110244-Post.gz deleted.\n    db-bkp-20060629054914-User.gz deleted.\n    db-bkp-20060629054914-Post.gz deleted.\n    db-bkp-20060329020048-User.gz deleted.\n    db-bkp-20060329020048-Post.gz deleted.\n    db-bkp-20050324012859-User.gz deleted.\n    db-bkp-20050324012859-Post.gz deleted.\n```\n\n## Requirements & Dependencies\n\n**AlchemyDumps** is tested and should work with Python 3.6+.\n\n## Tests\n\nIf you wanna run the tests for the current Python version:\n\n```console\npoetry install\npoetry run nosetests\n```\n\nWe also use some style and quality checkers:\n\n```console\npoetry run black . --check\npoetry run flake8 flask_alchemydumps/ tests/\n```\n\nIf you wanna cover all supported Python version, you need them installed and available via [`pyenv`](https://github.com/pyenv/pyenv). Then just `poetry run tox`.',
    'author': 'Eduardo Cuducos',
    'author_email': 'cuducos@users.noreply.github.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/cuducos/alchemydumps',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
