/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.dboe.transaction.txn;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Consumer;
import org.apache.jena.atlas.logging.FmtLog;
import org.apache.jena.atlas.logging.Log;
import org.apache.jena.dboe.base.file.FileException;
import org.apache.jena.dboe.base.file.Location;
import org.apache.jena.dboe.sys.SysDB;
import org.apache.jena.dboe.transaction.txn.ComponentGroup;
import org.apache.jena.dboe.transaction.txn.ComponentId;
import org.apache.jena.dboe.transaction.txn.PrepareState;
import org.apache.jena.dboe.transaction.txn.QuorumGenerator;
import org.apache.jena.dboe.transaction.txn.SysTrans;
import org.apache.jena.dboe.transaction.txn.SysTransState;
import org.apache.jena.dboe.transaction.txn.Transaction;
import org.apache.jena.dboe.transaction.txn.TransactionCoordinatorState;
import org.apache.jena.dboe.transaction.txn.TransactionException;
import org.apache.jena.dboe.transaction.txn.TransactionListener;
import org.apache.jena.dboe.transaction.txn.TransactionalComponent;
import org.apache.jena.dboe.transaction.txn.TxnId;
import org.apache.jena.dboe.transaction.txn.TxnIdFactory;
import org.apache.jena.dboe.transaction.txn.TxnIdGenerator;
import org.apache.jena.dboe.transaction.txn.TxnState;
import org.apache.jena.dboe.transaction.txn.journal.Journal;
import org.apache.jena.dboe.transaction.txn.journal.JournalEntry;
import org.apache.jena.dboe.transaction.txn.journal.JournalEntryType;
import org.apache.jena.query.ReadWrite;
import org.apache.jena.query.TxnType;
import org.slf4j.Logger;

public final class TransactionCoordinator {
    private static Logger SysLog = SysDB.syslog;
    private static Logger SysErr = SysDB.errlog;
    private final Journal journal;
    private boolean configurable = true;
    private final ComponentGroup components = new ComponentGroup(new TransactionalComponent[0]);
    private final List<TransactionListener> listeners = new ArrayList<TransactionListener>();
    private List<ShutdownHook> shutdownHooks;
    private TxnIdGenerator txnIdGenerator = TxnIdFactory.txnIdGenSimple;
    private QuorumGenerator quorumGenerator = null;
    private Semaphore writersWaiting = new Semaphore(1, true);
    private ReadWriteLock exclusivitylock = new ReentrantReadWriteLock();
    private final AtomicLong dataVersion = new AtomicLong(0L);
    private Object coordinatorLock = new Object();
    private static final boolean promotionWaitForWriters = true;
    private Set<Transaction> activeTransactions = ConcurrentHashMap.newKeySet();
    private AtomicLong activeTransactionCount = new AtomicLong(0L);
    private AtomicLong activeReadersCount = new AtomicLong(0L);
    private AtomicLong activeWritersCount = new AtomicLong(0L);
    private final AtomicLong countBegin = new AtomicLong(0L);
    private final AtomicLong countBeginRead = new AtomicLong(0L);
    private final AtomicLong countBeginWrite = new AtomicLong(0L);
    private final AtomicLong countFinished = new AtomicLong(0L);

    public TransactionCoordinator(Location location) {
        this(Journal.create(location));
    }

    public TransactionCoordinator(Journal journal) {
        this(journal, null, new ArrayList<ShutdownHook>());
    }

    public TransactionCoordinator(Journal journal, List<TransactionalComponent> components) {
        this(journal, components, new ArrayList<ShutdownHook>());
    }

    private TransactionCoordinator(Journal journal, List<TransactionalComponent> txnComp, List<ShutdownHook> shutdownHooks) {
        this.journal = journal;
        this.shutdownHooks = new ArrayList<ShutdownHook>(shutdownHooks);
        if (txnComp != null) {
            this.components.addAll(txnComp);
        }
    }

    public TransactionCoordinator add(TransactionalComponent elt) {
        this.checklAllowModification();
        this.components.add(elt);
        return this;
    }

    public TransactionCoordinator remove(TransactionalComponent elt) {
        this.checklAllowModification();
        this.components.remove(elt.getComponentId());
        return this;
    }

    public TransactionCoordinator addListener(TransactionListener listener) {
        this.checklAllowModification();
        this.listeners.add(listener);
        return this;
    }

    public TransactionCoordinator removeListener(TransactionListener listener) {
        this.checklAllowModification();
        this.listeners.remove(listener);
        return this;
    }

    public void modifyConfig(Runnable action) {
        try {
            this.startExclusiveMode();
            this.configurable = true;
            action.run();
        }
        finally {
            this.configurable = false;
            this.finishExclusiveMode();
        }
    }

    private void listeners(Consumer<TransactionListener> action) {
        this.listeners.forEach(x -> action.accept((TransactionListener)x));
    }

    public void add(ShutdownHook hook) {
        this.checklAllowModification();
        this.shutdownHooks.add(hook);
    }

    public void remove(ShutdownHook hook) {
        this.checklAllowModification();
        this.shutdownHooks.remove(hook);
    }

    public void setQuorumGenerator(QuorumGenerator qGen) {
        this.checklAllowModification();
        this.quorumGenerator = qGen;
    }

    public void start() {
        this.checklAllowModification();
        this.recovery();
        this.configurable = false;
    }

    private void recovery() {
        Iterator<JournalEntry> iter = this.journal.entries();
        if (!iter.hasNext()) {
            this.components.forEachComponent(c -> c.cleanStart());
            return;
        }
        SysLog.info("Journal recovery start");
        this.components.forEachComponent(c -> c.startRecovery());
        ArrayList entries = new ArrayList();
        iter.forEachRemaining(entry -> {
            switch (entry.getType()) {
                case ABORT: {
                    entries.clear();
                    break;
                }
                case COMMIT: {
                    this.recover(entries);
                    entries.clear();
                    break;
                }
                case REDO: 
                case UNDO: {
                    entries.add(entry);
                }
            }
        });
        this.components.forEachComponent(c -> c.finishRecovery());
        this.journal.reset();
        SysLog.info("Journal recovery end");
    }

    private void recover(List<JournalEntry> entries) {
        entries.forEach(e2 -> {
            if (e2.getType() == JournalEntryType.UNDO) {
                Log.warn(this, "UNDO entry : not handled");
                return;
            }
            ComponentId cid = e2.getComponentId();
            ByteBuffer bb = e2.getByteBuffer();
            TransactionalComponent c = this.components.findComponent(cid);
            if (c == null) {
                Log.warn(this, "No component for " + cid);
                return;
            }
            c.recover(bb);
        });
    }

    public void setTxnIdGenerator(TxnIdGenerator generator) {
        this.txnIdGenerator = generator;
    }

    public Journal getJournal() {
        return this.journal;
    }

    public Location getLocation() {
        return this.getJournal().getLocation();
    }

    public TransactionCoordinatorState detach(Transaction txn) {
        txn.detach();
        TransactionCoordinatorState coordinatorState = new TransactionCoordinatorState(txn);
        this.components.forEach((id, c) -> {
            SysTransState s = c.detach();
            coordinatorState.componentStates.put((ComponentId)id, s);
        });
        return coordinatorState;
    }

    public void attach(TransactionCoordinatorState coordinatorState) {
        Transaction txn = coordinatorState.transaction;
        txn.attach();
        coordinatorState.componentStates.forEach((id, obj) -> this.components.findComponent((ComponentId)id).attach((SysTransState)obj));
    }

    public void shutdown() {
        this.shutdown(false);
    }

    public void shutdown(boolean silent) {
        if (this.coordinatorLock == null) {
            return;
        }
        if (!silent && this.countActive() > 0L) {
            FmtLog.warn(SysErr, "Transactions active: W=%d, R=%d", this.countActiveWriter(), this.countActiveReaders());
        }
        this.components.forEach((id, c) -> c.shutdown());
        this.shutdownHooks.forEach(h -> h.shutdown());
        this.coordinatorLock = null;
        this.journal.close();
    }

    private void checklAllowModification() {
        if (!this.configurable) {
            throw new TransactionException("TransactionCoordinator configuration is locked");
        }
    }

    private void checkActive() {
        if (this.configurable) {
            throw new TransactionException("TransactionCoordinator has not been started");
        }
        this.checkNotShutdown();
    }

    private void checkNotShutdown() {
        if (this.coordinatorLock == null) {
            throw new TransactionException("TransactionCoordinator has been shutdown");
        }
    }

    private void releaseWriterLock() {
        int x = this.writersWaiting.availablePermits();
        if (x != 0) {
            throw new TransactionException("TransactionCoordinator: Probably mismatch of enable/disableWriter calls");
        }
        this.writersWaiting.release();
    }

    private boolean acquireWriterLock(boolean canBlock) {
        if (!canBlock) {
            return this.writersWaiting.tryAcquire();
        }
        try {
            this.writersWaiting.acquire();
            return true;
        }
        catch (InterruptedException e2) {
            throw new TransactionException(e2);
        }
    }

    public void startExclusiveMode() {
        this.startExclusiveMode(true);
    }

    public boolean tryExclusiveMode() {
        return this.tryExclusiveMode(false);
    }

    public boolean tryExclusiveMode(boolean canBlock) {
        return this.startExclusiveMode(canBlock);
    }

    private boolean startExclusiveMode(boolean canBlock) {
        if (canBlock) {
            this.exclusivitylock.writeLock().lock();
            return true;
        }
        return this.exclusivitylock.writeLock().tryLock();
    }

    public void finishExclusiveMode() {
        this.exclusivitylock.writeLock().unlock();
    }

    public void execExclusive(Runnable action) {
        this.startExclusiveMode();
        try {
            action.run();
        }
        finally {
            this.finishExclusiveMode();
        }
    }

    public void blockWriters() {
        this.acquireWriterLock(true);
    }

    public boolean tryBlockWriters() {
        return this.tryBlockWriters(false);
    }

    public boolean tryBlockWriters(boolean canBlock) {
        return this.acquireWriterLock(canBlock);
    }

    public void enableWriters() {
        this.releaseWriterLock();
    }

    public void execAsWriter(Runnable action) {
        this.blockWriters();
        try {
            action.run();
        }
        finally {
            this.enableWriters();
        }
    }

    public Transaction begin(TxnType txnType) {
        return this.begin(txnType, true);
    }

    public Transaction begin(TxnType txnType, boolean canBlock) {
        boolean b;
        Objects.nonNull((Object)txnType);
        this.checkActive();
        if (canBlock) {
            this.exclusivitylock.readLock().lock();
        } else if (!this.exclusivitylock.readLock().tryLock()) {
            return null;
        }
        if (txnType == TxnType.WRITE && !(b = this.acquireWriterLock(canBlock))) {
            this.exclusivitylock.readLock().unlock();
            return null;
        }
        Transaction transaction = this.begin$(txnType);
        this.startActiveTransaction(transaction);
        transaction.begin();
        this.notifyBegin(transaction);
        return transaction;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Transaction begin$(TxnType txnType) {
        Object object = this.coordinatorLock;
        synchronized (object) {
            this.checkActive();
            TxnId txnId = this.txnIdGenerator.generate();
            ArrayList<SysTrans> sysTransList = new ArrayList<SysTrans>();
            Transaction transaction = new Transaction(this, txnType, TransactionCoordinator.initialMode(txnType), txnId, this.dataVersion.get(), sysTransList);
            ComponentGroup txnComponents = this.chooseComponents(this.components, txnType);
            txnComponents.forEachComponent(elt -> {
                SysTrans sysTrans = new SysTrans((TransactionalComponent)elt, transaction, txnId);
                sysTransList.add(sysTrans);
            });
            txnComponents.forEachComponent(elt -> elt.begin(transaction));
            return transaction;
        }
    }

    private static ReadWrite initialMode(TxnType txnType) {
        return TxnType.initial(txnType);
    }

    private ComponentGroup chooseComponents(ComponentGroup components, TxnType txnType) {
        if (this.quorumGenerator == null) {
            return components;
        }
        ComponentGroup cg = this.quorumGenerator.genQuorum(txnType);
        if (cg == null) {
            return components;
        }
        cg.forEach((id, c) -> {
            TransactionalComponent tcx = components.findComponent((ComponentId)id);
            if (!tcx.equals(c)) {
                SysLog.warn("TransactionalComponent not in TransactionCoordinator's ComponentGroup");
            }
        });
        if (SysLog.isDebugEnabled()) {
            SysLog.debug("Custom ComponentGroup for transaction " + txnType + ": size=" + cg.size() + " of " + components.size());
        }
        return cg;
    }

    boolean executePromote(Transaction transaction, boolean readCommittedPromotion) {
        if (transaction.getMode() == ReadWrite.WRITE) {
            return true;
        }
        if (transaction.getTxnType() == TxnType.READ) {
            throw new TransactionException("promote: can't promote a READ transaction");
        }
        this.notifyPromoteStart(transaction);
        boolean b = this.promoteTxn$(transaction, readCommittedPromotion);
        this.notifyPromoteFinish(transaction);
        return b;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean promoteTxn$(Transaction transaction, boolean readCommittedPromotion) {
        if (transaction.getTxnType() == TxnType.READ_COMMITTED_PROMOTE) {
            if (!this.promotionWaitForWriters()) {
                return false;
            }
            Object object = this.coordinatorLock;
            synchronized (object) {
                try {
                    transaction.promoteComponents();
                }
                catch (TransactionException ex) {
                    try {
                        transaction.abort();
                    }
                    catch (RuntimeException runtimeException) {
                        // empty catch block
                    }
                    this.releaseWriterLock();
                    return false;
                }
                this.promoteActiveTransaction(transaction);
            }
            return true;
        }
        if (!this.checkNoInterveningCommits(transaction)) {
            return false;
        }
        if (!this.promotionWaitForWriters()) {
            return false;
        }
        Object object = this.coordinatorLock;
        synchronized (object) {
            if (!this.checkNoInterveningCommits(transaction)) {
                this.releaseWriterLock();
                return false;
            }
            try {
                transaction.promoteComponents();
            }
            catch (TransactionException ex) {
                try {
                    transaction.abort();
                }
                catch (RuntimeException runtimeException) {
                    // empty catch block
                }
                this.releaseWriterLock();
                return false;
            }
            this.promoteActiveTransaction(transaction);
        }
        return true;
    }

    private boolean checkNoInterveningCommits(Transaction transaction) {
        long currentEpoch;
        long txnEpoch = transaction.getDataVersion();
        return txnEpoch >= (currentEpoch = this.dataVersion.get());
    }

    private boolean promotionWaitForWriters() {
        return this.acquireWriterLock(true);
    }

    void completed(Transaction transaction) {
        this.finishActiveTransaction(transaction);
        this.notifyEnd(transaction);
    }

    void executePrepare(Transaction transaction) {
        this.notifyPrepareStart(transaction);
        transaction.getComponents().forEach(sysTrans -> {
            ByteBuffer data = sysTrans.commitPrepare();
            if (data != null) {
                PrepareState s = new PrepareState(sysTrans.getComponentId(), data);
                this.journal.write(s);
            }
        });
        this.notifyPrepareFinish(transaction);
    }

    void executeCommit(Transaction transaction, Runnable commit, Runnable finish, Runnable sysabort) {
        this.notifyCommitStart(transaction);
        if (transaction.isReadTxn()) {
            finish.run();
            this.notifyCommitFinish(transaction);
            return;
        }
        this.journal.startWrite();
        try {
            this.executeCommitWriter(transaction, commit, finish, sysabort);
            this.journal.commitWrite();
        }
        catch (TransactionException ex) {
            throw ex;
        }
        catch (Throwable th) {
            throw th;
        }
        finally {
            this.journal.endWrite();
        }
        this.notifyCommitFinish(transaction);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void executeCommitWriter(Transaction transaction, Runnable commit, Runnable finish, Runnable sysabort) {
        Object object = this.coordinatorLock;
        synchronized (object) {
            try {
                this.journal.writeJournal(JournalEntry.COMMIT);
                this.journal.sync();
            }
            catch (FileException ex) {
                if (ex.getCause() instanceof ClosedByInterruptException) {
                    this.journal.reopen();
                    this.rollback(transaction, sysabort);
                    SysLog.warn("Thread interrupt during I/O in 'commit' : executed transaction rollback: " + ex.getMessage());
                    throw new TransactionException("Thread interrupt during I/O in 'commit' : transaction rollback.", ex);
                }
                if (this.isIOException(ex)) {
                    SysErr.warn("IOException during 'commit' : transaction may have committed. Attempting rollback: " + ex.getMessage());
                } else {
                    SysErr.warn("Exception during 'commit' : transaction may have committed. Attempting rollback. Details:", ex);
                }
                if (this.abandonTxn(transaction, sysabort)) {
                    SysErr.warn("Transaction rollback");
                    throw new TransactionException("Exception during 'commit' - transaction rollback.", ex);
                }
                SysErr.error("Transaction rollback failed. System unstable.\nPlease contact users@jena.apache.org, giving details of the environment and this incident.");
                throw new Error("Exception during 'rollback' - System unstable.", ex);
            }
            catch (Throwable ex) {
                SysErr.warn("Unexpected Throwable during 'commit' : transaction may have committed. Attempting rollback: ", ex);
                if (this.abandonTxn(transaction, sysabort)) {
                    SysErr.warn("Transaction rollback");
                    throw new TransactionException("Exception during 'commit' - transaction rollback.", ex);
                }
                SysErr.error("Transaction rollback failed. System unstable.");
                throw new TransactionException("Exception during 'rollback' - System unstable.", ex);
            }
            commit.run();
            this.journal.truncate(0L);
            finish.run();
            this.advanceDataVersion();
        }
    }

    private void advanceDataVersion() {
        this.dataVersion.incrementAndGet();
    }

    private void abandonIfInterruped(Transaction txn, Runnable sysabort, String msg) {
        if (Thread.interrupted()) {
            this.abandonTxn(txn, sysabort);
            Thread.currentThread().interrupt();
            throw new TransactionException(msg);
        }
    }

    private boolean abandonTxn(Transaction txn, Runnable sysabort) {
        try {
            this.journal.abortWrite();
            this.rollback(txn, sysabort);
            return true;
        }
        catch (Throwable th) {
            SysErr.warn("Exception during system rollback", th);
            return false;
        }
    }

    private void rollback(Transaction txn, Runnable sysabort) {
        txn.setState(TxnState.ACTIVE);
        sysabort.run();
        txn.setState(TxnState.ABORTED);
    }

    private boolean isIOException(Throwable ex) {
        while (ex != null) {
            if (ex instanceof IOException) {
                return true;
            }
            ex = ex.getCause();
        }
        return false;
    }

    void executeAbort(Transaction transaction, Runnable abort) {
        this.notifyAbortStart(transaction);
        abort.run();
        this.notifyAbortFinish(transaction);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void startActiveTransaction(Transaction transaction) {
        Object object = this.coordinatorLock;
        synchronized (object) {
            this.countBegin.incrementAndGet();
            switch (transaction.getMode()) {
                case READ: {
                    this.countBeginRead.incrementAndGet();
                    this.activeReadersCount.incrementAndGet();
                    break;
                }
                case WRITE: {
                    this.countBeginWrite.incrementAndGet();
                    this.activeWritersCount.incrementAndGet();
                }
            }
            this.activeTransactionCount.incrementAndGet();
            this.activeTransactions.add(transaction);
        }
    }

    private void promoteActiveTransaction(Transaction transaction) {
        this.activeReadersCount.decrementAndGet();
        this.activeWritersCount.incrementAndGet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void finishActiveTransaction(Transaction transaction) {
        Object object = this.coordinatorLock;
        synchronized (object) {
            boolean x = this.activeTransactions.remove(transaction);
            if (!x) {
                return;
            }
            this.countFinished.incrementAndGet();
            this.activeTransactionCount.decrementAndGet();
            switch (transaction.getMode()) {
                case READ: {
                    this.activeReadersCount.decrementAndGet();
                    break;
                }
                case WRITE: {
                    this.activeWritersCount.decrementAndGet();
                }
            }
        }
        this.exclusivitylock.readLock().unlock();
    }

    public long countActiveReaders() {
        return this.activeReadersCount.get();
    }

    public long countActiveWriter() {
        return this.activeWritersCount.get();
    }

    public long countActive() {
        return this.activeTransactionCount.get();
    }

    private void notifyBegin(Transaction transaction) {
        this.listeners(x -> x.notifyTxnStart(transaction));
    }

    private void notifyEnd(Transaction transaction) {
        this.listeners(x -> x.notifyTxnFinish(transaction));
    }

    private void notifyPromoteStart(Transaction transaction) {
        this.listeners(x -> x.notifyPromoteStart(transaction));
    }

    private void notifyPromoteFinish(Transaction transaction) {
        this.listeners(x -> x.notifyPromoteFinish(transaction));
    }

    private void notifyPrepareStart(Transaction transaction) {
        this.listeners(x -> x.notifyPrepareStart(transaction));
    }

    private void notifyPrepareFinish(Transaction transaction) {
        this.listeners(x -> x.notifyPrepareFinish(transaction));
    }

    private void notifyCommitStart(Transaction transaction) {
        this.listeners(x -> x.notifyCommitStart(transaction));
    }

    private void notifyCommitFinish(Transaction transaction) {
        this.listeners(x -> x.notifyCommitFinish(transaction));
        if (transaction.getMode() == ReadWrite.WRITE) {
            this.releaseWriterLock();
        }
    }

    private void notifyAbortStart(Transaction transaction) {
        this.listeners(x -> x.notifyAbortStart(transaction));
    }

    private void notifyAbortFinish(Transaction transaction) {
        this.listeners(x -> x.notifyAbortFinish(transaction));
        if (transaction.getMode() == ReadWrite.WRITE) {
            this.releaseWriterLock();
        }
    }

    void notifyEndStart(Transaction transaction) {
        this.listeners(x -> x.notifyEndStart(transaction));
    }

    void notifyEndFinish(Transaction transaction) {
        this.listeners(x -> x.notifyEndFinish(transaction));
    }

    void notifyCompleteStart(Transaction transaction) {
        this.listeners(x -> x.notifyCompleteStart(transaction));
    }

    void notifyCompleteFinish(Transaction transaction) {
        this.listeners(x -> x.notifyCompleteFinish(transaction));
    }

    public long countBegin() {
        return this.countBegin.get();
    }

    public long countBeginRead() {
        return this.countBeginRead.get();
    }

    public long countBeginWrite() {
        return this.countBeginWrite.get();
    }

    public long countFinished() {
        return this.countFinished.get();
    }

    @FunctionalInterface
    public static interface ShutdownHook {
        public void shutdown();
    }
}

