"""Solving a single transcription of an OCP to an NLP."""


import collections
import csv
import itertools
import json
from timeit import default_timer as timer

import matplotlib.pyplot as plt
import numba as nb
import numpy as np
import scipy.integrate as integrate
import scipy.interpolate as interpolate
import scipy.sparse as sparse
import sympy as sym
from pyproprop import processed_property

from .guess import (PhaseGuess, EndpointGuess, Guess)
from .mesh import Mesh
from .nlp import initialise_nlp_backend
from .scaling import IterationScaling
from .utils import console_out, format_time


class Iteration:
    """A single mesh iteration (OCP transcription to an NLP)."""

    backend = processed_property("backend", read_only=True)
    ocp = processed_property("ocp", read_only=True)
    mesh = processed_property("mesh", read_only=True)
    solution = processed_property("solution", read_only=True)
    index = processed_property("index", read_only=True)
    number = processed_property("number", read_only=True)
    solved = processed_property("solved", type=bool)

    def __init__(self, backend, index, mesh, guess):
        """Initialise the iteration object.

        Handles full instantiation of the iteration object so that as soon as
        the solve method is called it can be handed off directly to the NLP
        solver.

        Parameters
        ----------
        backend : pycollo.Backend
            The backend associated with this OCP.
        index : int
            Index for the iteration. 0 corresponds for first iteration so that
            this attribute can be directly used to index in to iterables of
            iterations etc.
        mesh : pycollo.Mesh
            The temporal discretised mesh over which this iteration will be
            solved.
        guess : pycollo.Guess
            The guess from the previous iteration, or the initial guess for the
            first iteration.

        """
        self.backend = backend
        self.ocp = self.backend.ocp
        self.index = index
        self.number = index + 1
        self.mesh = mesh
        self.prev_guess = guess
        self.initialise()

    @property
    def optimal_control_problem(self):
        """Alias to `self.ocp`."""
        return self.ocp

    @property
    def solved(self):
        return hasattr(self, "_solution")

    def initialise(self):
        """Abstraction layer for all steps in initialising iteration."""
        self.console_out_initialising_iteration()
        self.interpolate_guess_to_mesh(self.prev_guess)
        self.create_variable_constraint_counts_slices()
        self.initialise_scaling()
        self.scale_guess()
        self.generate_nlp()
        self.generate_bounds()
        self.check_nlp_functions()
        self.console_out_iteration_initialised()

    def console_out_initialising_iteration(self):
        """Console out message stating iteration initialisation started."""
        msg = f"Initialising mesh iteration #{self.number}."
        console_out(msg, heading=True)

    def interpolate_guess_to_mesh(self, prev_guess):
        """Interpolate the previous guess to the new mesh.

        Parameters
        ----------
        prev_guess : pycollo.Guess
            The guess at the previous mesh iteration, or for the first
            iteration, the initial guess generated from user-supplied info.

        """
        def interpolate_to_new_mesh(prev_tau, tau, num_vars, prev, N):
            """Iterpolate previous mesh to new mesh.

            Linear interpolation is used. Bounding errors in the interpolation
            are allowed to accomdate for small numerical floating point errors.
            This should never cause a problem as the data used here is
            generated by the Pycollo backend and should be correctly processed.
            Fill values are extrapolated again to accommodate small floating
            point errors.

            Parameters
            ----------
            prev_tau : np.ndarray
                The nondimensionalised temporal discretisation of the previous
                iteration mesh.
            tau : np.ndarray
                The nondimensionalised temporal discretisation of the new mesh.
            num_vars : int
                Number of a specific type of variable being interpolated.
            prev : np.ndarray
                The guess on the previous mesh discretisation.
            N : int
                Number of temporal nodes in the new discretisation.

            Returns
            -------
            np.ndarray
                The iterpolated guess for a specific variable category.

            Notes
            -----
            Replace `interpolate.interp1d(**kwargs) with a functools.partial().

            """
            new_guess = np.empty((num_vars, N))
            for index, row in enumerate(prev):
                interp_func = interpolate.interp1d(prev_tau,
                                                   row,
                                                   bounds_error=False,
                                                   fill_value="extrapolate")
                new_guess[index, :] = interp_func(tau)
            return new_guess

        time_guess_start = timer()
        self.guess_tau = self.mesh.tau
        self.guess_t0 = prev_guess.t0
        self.guess_tF = prev_guess.tF
        self.guess_stretch = [0.5 * (tF - t0)
                              for t0, tF in zip(self.guess_t0, self.guess_tF)]
        self.guess_shift = [0.5 * (t0 + tF)
                            for t0, tF in zip(self.guess_t0, self.guess_tF)]
        zipped = zip(self.guess_tau, self.guess_stretch, self.guess_shift)
        self.guess_time = [tau * stretch + shift
                           for tau, stretch, shift in zipped]
        zipped = zip(prev_guess.tau,
                     self.guess_tau,
                     self.backend.p,
                     prev_guess.y,
                     self.mesh.N)
        self.guess_y = [interpolate_to_new_mesh(prev_tau,
                                                tau,
                                                p.num_y_var,
                                                prev_y,
                                                N)
                        for prev_tau, tau, p, prev_y, N in zipped]
        zipped = zip(prev_guess.tau,
                     self.guess_tau,
                     self.backend.p,
                     prev_guess.u,
                     self.mesh.N)
        self.guess_u = [interpolate_to_new_mesh(prev_tau,
                                                tau,
                                                p.num_u_var,
                                                prev_u,
                                                N)
                        for prev_tau, tau, p, prev_u, N in zipped]
        self.guess_q = prev_guess.q
        self.guess_t = prev_guess.t
        self.guess_s = prev_guess.s

        self.guess_x = []
        zipped = zip(self.guess_y,
                     self.guess_u,
                     self.guess_q,
                     self.guess_t)
        for y, u, q, t in zipped:
            self.guess_x.extend(y.tolist())
            self.guess_x.extend(u.tolist())
            self.guess_x.extend([q.tolist()])
            self.guess_x.extend([t.tolist()])
        self.guess_x.extend([self.guess_s.tolist()])
        guess_x_chained = itertools.chain.from_iterable(self.guess_x)
        self.guess_x = np.array(list(guess_x_chained))

        time_guess_stop = timer()
        self._time_guess_interpolation = time_guess_stop - time_guess_start
        msg = (f"Guess interpolated to iteration mesh in "
               f"{format_time(self._time_guess_interpolation)}.")
        console_out(msg)

    def create_variable_constraint_counts_slices(self):
        """Abstraction layer for slices of variables and constraints.

        Slices are needed and heavilty used for easy indexing.

        """
        self.create_variable_counts_per_phase_and_total()
        self.create_variable_slices_per_phase()
        self.create_constraint_counts_per_phase_and_total()
        self.create_constraint_slices_per_phase()
        self.create_constraint_component_function_slices_per_phase()

    def create_variable_counts_per_phase_and_total(self):
        """Counts of all variable categories per phase and overall totals."""
        self.num_y_per_phase = []
        self.num_u_per_phase = []
        self.num_q_per_phase = []
        self.num_t_per_phase = []
        self.num_x_per_phase = []
        for p_backend, N in zip(self.backend.p, self.mesh.N):
            num_y = p_backend.num_y_var * N
            num_u = p_backend.num_u_var * N
            num_q = p_backend.num_q_var
            num_t = p_backend.num_t_var
            num_x = num_y + num_u + num_q + num_t
            self.num_y_per_phase.append(num_y)
            self.num_u_per_phase.append(num_u)
            self.num_q_per_phase.append(num_q)
            self.num_t_per_phase.append(num_t)
            self.num_x_per_phase.append(num_x)
        self.num_y = sum(num_y for num_y in self.num_y_per_phase)
        self.num_u = sum(num_u for num_u in self.num_u_per_phase)
        self.num_q = sum(num_q for num_q in self.num_q_per_phase)
        self.num_t = sum(num_t for num_t in self.num_t_per_phase)
        self.num_s = self.backend.num_s_var
        self.num_x = sum([self.num_y,
                          self.num_u,
                          self.num_q,
                          self.num_t,
                          self.num_s])

    def create_variable_slices_per_phase(self):
        """Slices to each variable category per phase from all variables."""
        self.y_slices = []
        self.u_slices = []
        self.q_slices = []
        self.t_slices = []
        self.x_slices = []
        total = 0
        zipped = zip(self.num_y_per_phase,
                     self.num_u_per_phase,
                     self.num_q_per_phase,
                     self.num_t_per_phase,
                     self.num_x_per_phase)
        for num_y, num_u, num_q, num_t, num_x in zipped:
            y_slice = slice(total, total + num_y)
            u_slice = slice(y_slice.stop, y_slice.stop + num_u)
            q_slice = slice(u_slice.stop, u_slice.stop + num_q)
            t_slice = slice(q_slice.stop, q_slice.stop + num_t)
            x_slice = slice(y_slice.start, t_slice.stop)
            self.y_slices.append(y_slice)
            self.u_slices.append(u_slice)
            self.q_slices.append(q_slice)
            self.t_slices.append(t_slice)
            self.x_slices.append(x_slice)
            total += num_x
        self.s_slice = slice(self.num_x - self.num_s, self.num_x)

    def create_constraint_counts_per_phase_and_total(self):
        """Counts of all constraint categories per phase and overall totals."""
        self.num_c_defect_per_phase = []
        self.num_c_path_per_phase = []
        self.num_c_integral_per_phase = []
        self.num_c_per_phase = []
        zipped = zip(self.backend.p, self.mesh.N, self.mesh.num_c_defect_per_y)
        for p_backend, N, num_c_defect_per_y in zipped:
            num_c_defect = p_backend.num_y_eqn * num_c_defect_per_y
            num_c_path = p_backend.num_p_con * N
            num_c_integral = p_backend.num_q_fnc
            num_c = num_c_defect + num_c_path + num_c_integral
            self.num_c_defect_per_phase.append(num_c_defect)
            self.num_c_path_per_phase.append(num_c_path)
            self.num_c_integral_per_phase.append(num_c_integral)
            self.num_c_per_phase.append(num_c)
        self.num_c_defect = sum(num_c
                                for num_c in self.num_c_defect_per_phase)
        self.num_c_path = sum(num_c
                              for num_c in self.num_c_path_per_phase)
        self.num_c_integral = sum(num_c
                                  for num_c in self.num_c_integral_per_phase)
        self.num_c_endpoint = self.backend.num_b_con
        self.num_c = sum([self.num_c_defect,
                          self.num_c_path,
                          self.num_c_integral,
                          self.num_c_endpoint])

    def create_constraint_slices_per_phase(self):
        """Slices to each variable category per phase for all constraints."""
        self.c_defect_slices = []
        self.c_path_slices = []
        self.c_integral_slices = []
        self.c_slices = []
        total = 0
        zipped = zip(self.num_c_defect_per_phase,
                     self.num_c_path_per_phase,
                     self.num_c_integral_per_phase,
                     self.num_c_per_phase)
        for num_c_defect, num_c_path, num_c_integral, num_c in zipped:
            c_defect_start = total
            c_path_start = c_defect_start + num_c_defect
            c_integral_start = c_path_start + num_c_path
            c_con_stop = c_integral_start + num_c_integral
            self.c_defect_slices.append(slice(c_defect_start, c_path_start))
            self.c_path_slices.append(slice(c_path_start, c_integral_start))
            self.c_integral_slices.append(slice(c_integral_start, c_con_stop))
            self.c_slices.append(slice(c_defect_start, c_con_stop))
            total += num_c
        self.c_endpoint_slice = slice(self.num_c - self.num_c_endpoint,
                                      self.num_c)

    def create_constraint_component_function_slices_per_phase(self):
        """Slices to different types of constraint component functions.

        Constraint component functions are the dynamical state equations
        (`y_eqn`), path constraint equations (`p_con`) and integrand functions
        (`q_fnc`).

        """
        self.dy_slices = []
        self.y_eqn_slices = []
        self.p_con_slices = []
        self.q_fnc_slices = []
        zipped = zip(self.num_y_per_phase,
                     self.num_c_path_per_phase,
                     self.num_c_integral_per_phase,
                     self.mesh.N)
        dy_total = 0
        for num_y, num_c_path, num_c_integral, N in zipped:
            y_eqn_start = 0
            p_con_start = y_eqn_start + num_y
            q_fnc_start = p_con_start + num_c_path
            c_con_stop = q_fnc_start + num_c_integral * N
            self.dy_slices.append(slice(dy_total, dy_total + num_y))
            self.y_eqn_slices.append(slice(y_eqn_start, p_con_start))
            self.p_con_slices.append(slice(p_con_start, q_fnc_start))
            self.q_fnc_slices.append(slice(q_fnc_start, c_con_stop))
            dy_total += num_y

    def initialise_scaling(self):
        """Initialise iteration-specific scaling.

        Create and initialise the :py:class:`IterationScaling` object with
        scaling variables expanded to cover the specific mesh for this
        iteration.

        """
        init_scaling_start = timer()
        self.scaling = self.backend.iteration_scaling(self)
        init_scaling_stop = timer()
        self._time_initialise_scaling = init_scaling_stop - init_scaling_start
        msg = (f"Scaling initialised in "
               f"{format_time(self._time_initialise_scaling)}.")
        console_out(msg)

    def scale_guess(self):
        """Scale guess from x to x-tilde if first iteration.

        Only the initial iteration requires this scaling of the guess as for
        subsequent mesh iterations the guess comes from the solution on the
        previous mesh which is already in the scaled basis (i.e. x-tilde).

        """
        scale_guess_start = timer()
        self.guess_x = self.scaling.scale_x(self.guess_x)
        scale_guess_stop = timer()
        self._time_scale_guess = scale_guess_stop - scale_guess_start
        msg = f"Initial guess scaled in {format_time(self._time_scale_guess)}."
        console_out(msg)

    def generate_nlp(self):
        """Generate the NLP and all required components (backend-specific)."""
        gen_nlp_start = timer()
        self.backend.generate_nlp_function_callables(self)
        self.generate_scaling()
        self.backend.create_nlp_solver()
        gen_nlp_stop = timer()
        self._time_generate_nlp = gen_nlp_stop - gen_nlp_start
        msg = f"NLP generated in {format_time(self._time_generate_nlp)}."
        console_out(msg)

    def generate_scaling(self):
        """Generate objective function and constraint scaling."""
        gen_scaling_start = timer()
        self.scaling.generate_J_c_scaling()
        gen_scaling_stop = timer()
        self._time_generate_scaling = gen_scaling_stop - gen_scaling_start
        msg = (f"Scaling generated in "
               f"{format_time(self._time_generate_scaling)}.")
        console_out(msg)

    def generate_bounds(self):
        """Generate bounds for the mesh iteration NLP."""
        gen_bounds_start = timer()
        self.generate_variable_bounds()
        self.generate_constraint_bounds()
        self.scale_bounds()
        gen_bounds_stop = timer()
        self._time_generate_bounds = gen_bounds_stop - gen_bounds_start
        msg = (f"Mesh-specific bounds generated in "
               f"{format_time(self._time_generate_bounds)}.")
        console_out(msg, trailing_blank_line=True)

    def generate_variable_bounds(self):
        """Generate bounds for the NLP variables."""
        bnds = []
        for p, N in zip(self.backend.p, self.mesh.N):
            p_bnds = p.ocp_phase.bounds
            y_bnds = p_bnds._y_bnd[p_bnds._y_needed]
            y_t0_bnds = p_bnds._y_t0_bnd[p_bnds._y_needed]
            y_tF_bnds = p_bnds._y_tF_bnd[p_bnds._y_needed]
            u_bnds = p_bnds._u_bnd[p_bnds._u_needed]
            q_bnds = p_bnds._q_bnd[p_bnds._q_needed]
            t_bnds = p_bnds._t_bnd[p_bnds._t_needed]
            for y_bnd, y_t0_bnd, y_tF_bnd in zip(y_bnds, y_t0_bnds, y_tF_bnds):
                bnds.extend([y_t0_bnd] + [y_bnd] * (N - 2) + [y_tF_bnd])
            for u_bnd in u_bnds:
                bnds.extend([u_bnd] * N)
            bnds.extend(q_bnds)
            bnds.extend(t_bnds)
        s_bnds = self.ocp.bounds._s_bnd[self.ocp.bounds._s_needed]
        bnds.extend(s_bnds)
        bnds = np.array(bnds)
        self.x_bnd_l = bnds[:, 0]
        self.x_bnd_u = bnds[:, 1]

    def generate_constraint_bounds(self):
        """Generate bounds for the NLP constraints."""
        bnds = []
        zipped = zip(self.backend.p,
                     self.mesh.N,
                     self.num_c_defect_per_phase,
                     self.num_c_integral_per_phase)
        for p, N, num_c_defect, num_c_integral in zipped:
            bnds.extend([np.array((0, 0))] * num_c_defect)
            for c_bnd in p.ocp_phase.bounds._p_con_bnd:
                bnds.extend([c_bnd] * N)
            bnds.extend([np.array((0, 0))] * num_c_integral)
        bnds.extend(self.backend.ocp.bounds._b_con_bnd)
        bnds = np.array(bnds)
        self.c_bnd_l = bnds[:, 0]
        self.c_bnd_u = bnds[:, 1]

    def scale_bounds(self):
        """Scale bounds from user basis to tilde basis."""
        self.x_bnd_l = self.scaling.scale_x(self.x_bnd_l)
        self.x_bnd_u = self.scaling.scale_x(self.x_bnd_u)
        self.c_bnd_l = self.scaling.scale_c(self.c_bnd_l)
        self.c_bnd_u = self.scaling.scale_c(self.c_bnd_u)

    def check_nlp_functions(self):
        """Dumps values of the NLP callables evaluated at the initial guess."""
        if self.backend.ocp.settings.check_nlp_functions:
            raise NotImplementedError

    def console_out_iteration_initialised(self):
        """Console out message stating iteration initialised."""
        all_times = (self._time_guess_interpolation,
                     self._time_generate_scaling,
                     self._time_initialise_scaling,
                     self._time_generate_nlp,
                     self._time_scale_guess,
                     self._time_generate_bounds,
                     )
        self._time_iteration_initialisation = sum(all_times)
        msg = (f"Mesh iteration #{self.number} initialised in "
               f"{format_time(self._time_iteration_initialisation)}.")
        console_out(msg, trailing_blank_line=True)

    def solve(self):
        """Abstraction layer for solving the NLP and processing its solution.

        Returns
        -------
        bool
            Whether the mesh tolerance has been met or not.
        Mesh
            Refined mesh for the next mesh iteration.
        Guess
            Guess to be used for the next mesh iteration.

        """
        self.console_out_solving_iteration()
        nlp_result = self.solve_nlp()
        mesh_iteration_result = self.process_nlp_solution(nlp_result)
        return mesh_iteration_result

    def console_out_solving_iteration(self):
        """Console out message stating iteration solving started."""
        msg = f"Solving mesh iteration #{self.number}."
        console_out(msg, heading=True)

    def solve_nlp(self):
        """Solve the NLP."""
        solve_time_start = timer()
        nlp_result = self.backend.solve_nlp()
        solve_time_stop = timer()
        self._time_solve = solve_time_stop - solve_time_start
        return nlp_result

    def process_nlp_solution(self, nlp_result):
        msg = f"Post-processing mesh iteration #{self.number}."
        console_out(msg, heading=True)

        process_time_start = timer()
        self.solution = self.backend.process_solution(self, nlp_result)
        next_iter_mesh = self.solution.refine_mesh()
        next_iter_guess = self.generate_guess_for_next_mesh_iteration()
        process_time_stop = timer()
        self._time_process = process_time_stop - process_time_start
        msg = (f"Mesh iteration #{self.number} solved in "
               f"{format_time(self._time_solve)}.")
        console_out(msg)
        msg = (f"Mesh iteration #{self.number} post-processed in "
               f"{format_time(self._time_process)}.")
        console_out(msg, trailing_blank_line=True)
        mesh_tolerance_met = self.check_if_mesh_tolerance_met(next_iter_mesh)
        self.display_mesh_iteration_info(mesh_tolerance_met, next_iter_mesh)
        mesh_iteration_result = MeshIterationResult(mesh_tolerance_met,
                                                    next_iter_mesh,
                                                    next_iter_guess)
        return mesh_iteration_result

    def generate_guess_for_next_mesh_iteration(self):
        """Abstraction collecting and combining phase and problem guesses.

        Returns
        -------
        Guess
            The guess for the next mesh iteration.
        """
        phase_guesses = self.collect_next_mesh_iteration_phase_guesses()
        endpoint_guess = self.collect_next_mesh_iteration_endpoint_guess()
        next_guess = Guess(self.backend, phase_guesses, endpoint_guess)
        return next_guess

    def collect_next_mesh_iteration_phase_guesses(self):
        """Summary

        Returns
        -------
        TYPE
            Description
        """
        phase_guesses = [self.collect_single_next_mesh_iteration_phase_guess(p)
                         for p in self.backend.p]
        return phase_guesses

    def collect_single_next_mesh_iteration_phase_guess(self, phase_backend):
        """Summary

        Parameters
        ----------
        phase_backend : TYPE
            Description

        Returns
        -------
        TYPE
            Description
        """
        phase_guess = PhaseGuess(phase_backend)
        phase_guess.time = self.solution._time_[phase_backend.i]
        phase_guess.state_variables = self.solution._y[phase_backend.i]
        phase_guess.control_variables = self.solution._u[phase_backend.i]
        phase_guess.integral_variables = self.solution._q[phase_backend.i]
        return phase_guess

    def collect_next_mesh_iteration_endpoint_guess(self):
        """Summary

        Returns
        -------
        TYPE
            Description
        """
        endpoint_guess = EndpointGuess(self.optimal_control_problem)
        endpoint_guess.parameter_variables = self.solution._s
        return endpoint_guess

    def check_if_mesh_tolerance_met(self, next_iter_mesh):
        """Summary

        Parameters
        ----------
        next_iter_mesh : TYPE
            Description

        Returns
        -------
        TYPE
            Description
        """
        mesh_tol = self.optimal_control_problem.settings.mesh_tolerance
        error_over_max_mesh_tol = []
        mesh_refinement = self.solution.mesh_refinement
        for max_rel_mesh_errors in mesh_refinement.maximum_relative_mesh_errors:
            is_greater_than = np.max(max_rel_mesh_errors) > mesh_tol
            error_over_max_mesh_tol.append(is_greater_than)
        mesh_tol_met = not any(error_over_max_mesh_tol)
        return mesh_tol_met

    def display_mesh_iteration_info(self, mesh_tol_met, next_iter_mesh):
        """Summary

        Parameters
        ----------
        mesh_tol_met : TYPE
            Description
        next_iter_mesh : TYPE
            Description

        Raises
        ------
        NotImplementedError
            Description
        """
        msg = f"Analysing mesh iteration #{self.number}."
        console_out(msg, heading=True)

        max_rel_mesh_error = np.max(np.array([np.max(el)
            for el in self.solution.mesh_refinement.maximum_relative_mesh_errors]))

        print(f'Objective Evaluation:       {self.solution.objective}')
        print(f'Max Relative Mesh Error:    {max_rel_mesh_error}')
        print(f'Collocation Points Used:    {sum([N for N in self.mesh.N])}\n')
        if mesh_tol_met:
            print(f'Adjusting Collocation Mesh: {next_iter_mesh.K} mesh sections\n')

        self._time_complete = sum([self._time_iteration_initialisation,
                                   self._time_solve,
                                   self._time_process])
        msg = (f"Mesh iteration #{self.number} completed in "
               f"{format_time(self._time_complete)}.")
        console_out(msg, trailing_blank_line=True)

        settings = self.optimal_control_problem.settings
        if settings.display_mesh_result_info:
            raise NotImplementedError

        if settings.display_mesh_result_graph:
            self.solution.plot()


mesh_iteration_result_fields = ("mesh_tolerance_met",
                                "next_iteration_mesh",
                                "next_iteration_guess")
MeshIterationResult = collections.namedtuple("MeshIterationResult",
                                             mesh_iteration_result_fields)




















class IterationOld:

    def generate_nlp_lambdas(self):
        """Summary
        """
        self.generate_continuous_variables_reshape_lambda()
        self.generate_endpoint_variables_reshape_lambda()
        self.generate_objective_lambda()
        self.generate_gradient_lambda()
        self.generate_constraints_lambda()
        self.generate_jacobian_lambda()
        if self.ocp.settings.derivative_level == 2:
            self.generate_hessian_lambda()

    def generate_continuous_variables_reshape_lambda(self):
        """Summary
        """
        def reshape_x(x_tilde):
            """Summary
            
            Parameters
            ----------
            x_tilde : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            x = self.scaling.unscale_x(x_tilde)
            x_tuple = self.backend.compiled_functions.x_reshape_lambda(
                x, 
                self.y_slices, 
                self.u_slices, 
                self.q_slices, 
                self.t_slices, 
                self.s_slice,
                self.mesh.N,
                )
            return x_tuple

        self._reshape_x = reshape_x
        msg = "Continuous variable reshape lambda generated successfully."
        console_out(msg)

    def generate_endpoint_variables_reshape_lambda(self):
        """Summary
        """
        self._x_endpoint_indices = []
        for p_backend, x_slice, q_slice, t_slice, N in zip(self.backend.p, self.x_slices, self.q_slices, self.t_slices, self.mesh.N):
            for i in range(p_backend.num_y_vars):
                start = x_slice.start
                i_y_t0 = start + i*N
                i_y_tF = start + (i+1)*N - 1
                self._x_endpoint_indices.append(i_y_t0)
                self._x_endpoint_indices.append(i_y_tF)
            self._x_endpoint_indices.extend(list(range(q_slice.start, q_slice.stop)))
            self._x_endpoint_indices.extend(list(range(t_slice.start, t_slice.stop)))
        self._x_endpoint_indices.extend(list(range(self.s_slice.start, self.s_slice.stop)))

        def reshape_x_point(x_tilde):
            """Summary
            
            Parameters
            ----------
            x_tilde : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            x = self.scaling.unscale_x(x_tilde)
            return self.backend.compiled_functions.x_reshape_lambda_point(x, self._x_endpoint_indices)

        self._reshape_x_point = reshape_x_point
        msg = "Endpoint variable reshape lambda generated successfully."
        console_out(msg)

    def generate_objective_lambda(self):
        """Summary
        """
        def objective(x_tilde):
            """Summary
            
            Parameters
            ----------
            x_tilde : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            x_tuple_point = self._reshape_x_point(x_tilde)
            J = self.backend.compiled_functions.J_lambda(x_tuple_point)
            J_tilde = self.scaling.scale_J(J)
            # print(J)
            # print(J_tilde)
            return J_tilde

        self._objective_lambda = objective
        msg = "Objective function lambda generated successfully."
        console_out(msg)

    def generate_gradient_lambda(self):
        """Summary
        """
        def gradient(x):
            """Summary
            
            Parameters
            ----------
            x : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            x_tuple_point = self._reshape_x_point(x)
            g = self.backend.compiled_functions.g_lambda(x_tuple_point, self.mesh.N)
            g_tilde = self.scaling.scale_g(g)
            # print(g)
            # print(g_tilde)
            return g_tilde

        self._gradient_lambda = gradient
        msg = "Objective function gradient lambda generated successfully."
        console_out(msg)

    def generate_constraints_lambda(self):
        """Summary
        """
        def constraint(x):
            """Summary
            
            Parameters
            ----------
            x : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            x_tuple = self._reshape_x(x)
            x_tuple_point = self._reshape_x_point(x)
            c = self.backend.compiled_functions.c_lambda(
                x_tuple, 
                x_tuple_point, 
                self.mesh.sA_matrix, 
                self.mesh.sD_matrix, 
                self.mesh.W_matrix, 
                self.mesh.N,
                [slice(p_var_slice.start, p_var_slice.start + p.num_y_vars) for p, p_var_slice in zip(self.backend.p, self.backend.phase_variable_slices)],
                [slice(p_var_slice.start + p.num_y_vars + p.num_u_vars, p_var_slice.start + p.num_y_vars + p.num_u_vars + p.num_q_vars) for p, p_var_slice in zip(self.backend.p, self.backend.phase_variable_slices)],
                self.c_lambda_dy_slices,
                self.c_lambda_p_slices,
                self.c_lambda_g_slices,
                )
            c_tilde = self.scaling.scale_c(c)
            # print(c)
            # print(c_tilde)
            return c_tilde

        self._constraint_lambda = constraint
        msg = "Constraints lambda generated successfully."
        console_out(msg)

    def generate_jacobian_lambda(self):
        """Summary
        """
        def jacobian_data(x):
            """Summary
            
            Parameters
            ----------
            x : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            G = jacobian(x)
            G_zeros = sparse.coo_matrix((np.full(self._G_nnz, 1e-20), self._jacobian_structure_lambda()), shape=self._G_shape).tocsr()
            return (G + G_zeros).tocoo().data

        def jacobian(x):
            """Summary
            
            Parameters
            ----------
            x : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            x_tuple = self._reshape_x(x)
            x_tuple_point = self._reshape_x_point(x)
            G = self.backend.compiled_functions.G_lambda(
                self._G_shape,
                x_tuple, 
                x_tuple_point,
                self.mesh.sA_matrix, 
                self.mesh.sD_matrix, 
                self.mesh.W_matrix, 
                self.mesh.N,
                [c_slice.start for c_slice in self.c_slices],
                [x_slice.start for x_slice in self.x_slices],
                [slice(p_var_slice.start, p_var_slice.start + p.num_y_vars) for p, p_var_slice in zip(self.backend.p, self.backend.phase_variable_slices)],
                [slice(p_var_slice.start + p.num_y_vars, p_var_slice.start + p.num_y_vars + p.num_u_vars) for p, p_var_slice in zip(self.backend.p, self.backend.phase_variable_slices)],
                [slice(p_var_slice.start + p.num_y_vars + p.num_u_vars, p_var_slice.start + p.num_y_vars + p.num_u_vars + p.num_q_vars) for p, p_var_slice in zip(self.backend.p, self.backend.phase_variable_slices)],
                [slice(p_var_slice.start + p.num_y_vars + p.num_u_vars + p.num_q_vars, p_var_slice.start + p.num_y_vars + p.num_u_vars + p.num_q_vars + p.num_t_vars) for p, p_var_slice in zip(self.backend.p, self.backend.phase_variable_slices)],
                self.c_lambda_dy_slices,
                self.c_lambda_p_slices,
                self.c_lambda_g_slices,
                [slice(p_c_slice.start, p_c_slice.start + p.num_c_defect) for p, p_c_slice in zip(self.backend.p, self.backend.phase_constraint_slices)],
                [slice(p_c_slice.start + p.num_c_defect, p_c_slice.start + p.num_c_defect + p.num_c_path) for p, p_c_slice in zip(self.backend.p, self.backend.phase_constraint_slices)],
                [slice(p_c_slice.start + p.num_c_defect + p.num_c_path, p_c_slice.start + p.num_c_defect + p.num_c_path + p.num_c_integral) for p, p_c_slice in zip(self.backend.p, self.backend.phase_constraint_slices)],
                self.num_y_per_phase,
                self.num_u_per_phase,
                self.num_q_per_phase,
                self.num_t_per_phase,
                self.num_x_per_phase,
                self.backend.num_s_vars,
                self.num_c_defect_per_phase,
                self.num_c_path_per_phase,
                self.num_c_integral_per_phase,
                self.num_c_per_phase,
                self.num_c_endpoint,
                [p.y_slice for p in self.backend.p],
                [p.u_slice for p in self.backend.p],
                [p.q_slice for p in self.backend.p],
                [p.t_slice for p in self.backend.p],
                [p.c_defect_slice for p in self.backend.p],
                [p.c_path_slice for p in self.backend.p],
                [p.c_integral_slice for p in self.backend.p],
                )
            G_tilde = self.scaling.scale_G(G)
            # print(G)
            # print(G_tilde)
            return G_tilde

        self._G_shape = (self.num_c, self.num_x)
        x_sparsity_detect = np.full(self.num_x, np.nan)
        G_sparsity_detect = jacobian(x_sparsity_detect).tocoo()
        self._G_nonzero_row = G_sparsity_detect.row
        self._G_nonzero_col = G_sparsity_detect.col
        self._G_nnz = G_sparsity_detect.nnz

        self._jacobian_lambda = jacobian_data

        def jacobian_structure():
            """Summary
            
            Returns
            -------
            TYPE
                Description
            """
            return (self._G_nonzero_row, self._G_nonzero_col)

        self._jacobian_structure_lambda = jacobian_structure

        msg = "Jacobian of the constraints lambda generated successfully."
        console_out(msg)

    def generate_hessian_lambda(self):
        """Summary
        """
        def detect_hessian_sparsity():
            """Summary
            """
            self._sH_endpoint_indices = detect_endpoint_hessian_sparsity()
            self._sH_continuous_indices = detect_continuous_hessian_sparsity()

            x_sparsity_detect = np.full(self.num_x, np.nan)
            lagrange_sparsity_detect = np.full(self.num_c, np.nan)
            obj_factor_sparsity_detect = np.nan
            
            H_sparsity_detect = hessian(x_sparsity_detect, obj_factor_sparsity_detect, lagrange_sparsity_detect).tocoo()
            self._H_nonzero_row = H_sparsity_detect.row
            self._H_nonzero_col = H_sparsity_detect.col
            self._H_nnz = H_sparsity_detect.nnz

        def detect_endpoint_hessian_sparsity():
            """Summary
            
            Returns
            -------
            TYPE
                Description
            """
            def ocp_index_to_phase_index(ocp_index):
                """Summary
                
                Parameters
                ----------
                ocp_index : TYPE
                    Description
                
                Returns
                -------
                TYPE
                    Description
                """
                return ocp_index_to_phase_index_mapping[ocp_index]

            ocp_indices = list(range(self.backend.num_point_vars))
            phase_indices = []
            i = 0
            for p, N in zip(self.backend.p, self.mesh.N):
                for _ in range(p.num_y_vars):
                    phase_indices.append(i)
                    phase_indices.append(i + N - 1)
                    i += N
                for _ in range(p.num_u_vars):
                    i += N
                for _ in range(p.num_q_vars + p.num_t_vars):
                    phase_indices.append(i)
                    i += 1
            for _ in range(self.backend.num_s_vars):
                phase_indices.append(i)
                i += 1

            ocp_index_to_phase_index_mapping = dict(zip(ocp_indices, phase_indices))

            x_endpoint_row_indices = []
            x_endpoint_col_indices = []
            for i_row_ocp, i_col_ocp in self.backend.expression_graph.ddL_dxbdxb.entries.keys():
                i_row_phase = ocp_index_to_phase_index(i_row_ocp)
                i_col_phase = ocp_index_to_phase_index(i_col_ocp)
                x_endpoint_row_indices.append(i_row_phase)
                x_endpoint_col_indices.append(i_col_phase)
            return (x_endpoint_row_indices, x_endpoint_col_indices)

        def detect_continuous_hessian_sparsity():
            """Summary
            
            Returns
            -------
            TYPE
                Description
            """
            ocp_indices = []
            offset = 0
            for p in self.backend.p:
                phase_ocp_indices = [(i, j) for i in range(offset, offset + p.num_vars) for j in range(offset, i + 1)]

                offset += p.num_vars
                ocp_indices += phase_ocp_indices
            ocp_indices += [(i, j) for i in range(offset, offset + self.backend.num_vars) for j in range(i + 1)]
            phase_blocks = []
            endpoint_blocks = []
            for p, N in zip(self.backend.p, self.mesh.N):
                num_yu_ocp = p.num_y_vars + p.num_u_vars
                block_yu_yu = sparse.kron(np.tril(np.ones((num_yu_ocp, num_yu_ocp))), sparse.coo_matrix(([1], ([0], [0])), shape=(N, N)))
                num_qt_ocp = p.num_q_vars + p.num_t_vars 
                block_yu_qt = sparse.kron(np.ones((num_qt_ocp, num_yu_ocp)), sparse.coo_matrix(([2], ([0], [0])), shape=(1, N)))
                block_qt_qt = sparse.csr_matrix(3 * np.tril(np.ones((num_qt_ocp, num_qt_ocp))))
                block_yu_s = sparse.kron(np.ones((self.backend.num_s_vars, num_yu_ocp)), sparse.coo_matrix(([2], ([0], [0])), shape=(1, N)))
                block_qt_s = sparse.csr_matrix(3 * np.ones((self.backend.num_s_vars, num_qt_ocp)))
                phase_blocks.append(sparse.bmat([[block_yu_yu, None], [block_yu_qt, block_qt_qt]]))
                endpoint_blocks.append(sparse.hstack([block_yu_s, block_qt_s]))
            phase_block = sparse.block_diag(phase_blocks)
            endpoint_block = sparse.hstack(endpoint_blocks)
            parameter_block = sparse.csr_matrix(3 * np.tril(np.ones((self.backend.num_s_vars, self.backend.num_s_vars))))
            continuous = sparse.bmat([[phase_block, None], [endpoint_block, parameter_block]]).tocsr().tocoo()
            ocp_index_to_phase_index_mapping = dict(zip(ocp_indices, zip(zip(continuous.row, continuous.col), continuous.data)))
            H_continuous_indices_iteration = [ocp_index_to_phase_index_mapping[index] for index in self.backend.expression_graph.ddL_dxdx.entries]
            return H_continuous_indices_iteration

        def hessian_data(x, lagrange, obj_factor):
            """Summary
            
            Parameters
            ----------
            x : TYPE
                Description
            lagrange : TYPE
                Description
            obj_factor : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            H = hessian(x, obj_factor, lagrange)
            H_zeros = sparse.coo_matrix((np.full(self._H_nnz, 1e-20), self._hessian_structure_lambda()), shape=self._H_shape).tocsr()
            return (H + H_zeros).tocoo().data

        def reshape_lagrange(lagrange):
            """Summary
            
            Parameters
            ----------
            lagrange : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            lagrange = np.array(lagrange)
            lagrange_prime = self.scaling.scale_lagrange(lagrange)
            chunks = []
            for p, c_defect_slice, c_path_slice, c_integral_slice, sA_matrix, W_matrix in zip(self.backend.p, self.c_defect_slices, self.c_path_slices, self.c_integral_slices, self.mesh.sA_matrix, self.mesh.W_matrix):
                chunks.extend([*sA_matrix.T.dot(lagrange_prime[c_defect_slice].reshape((p.num_c_defect, -1)).T).T])
                if p.num_c_path:
                    chunks.extend([*lagrange_prime[c_path_slice].reshape((p.num_c_path, -1))])
                if p.num_c_integral:
                    chunks.extend([*lagrange_prime[c_integral_slice].reshape(-1, 1).dot(-W_matrix.reshape((1, -1)))])
            if self.num_c_endpoint:
                chunks.extend([*0*lagrange_prime[self.c_endpoint_slice].reshape(-1, )])

            return chunks

        def hessian(x, obj_factor, lagrange):
            """Summary
            
            Parameters
            ----------
            x : TYPE
                Description
            obj_factor : TYPE
                Description
            lagrange : TYPE
                Description
            
            Returns
            -------
            TYPE
                Description
            """
            x_tuple = self._reshape_x(x)
            x_tuple_point = self._reshape_x_point(x)
            obj_factor_prime = self.scaling.scale_sigma(obj_factor)
            lagrange_prime = reshape_lagrange(lagrange)
            H = self.backend.compiled_functions.H_lambda(
                self._H_shape,
                x_tuple, 
                x_tuple_point,
                obj_factor_prime,
                lagrange_prime,
                self._sH_continuous_indices,
                self._sH_endpoint_indices,
                )
            H_tilde = self.scaling.scale_H(H)
            return H_tilde

        self._H_shape = (self.num_x, self.num_x)
        detect_hessian_sparsity()
        self._hessian_lambda = hessian_data

        def hessian_structure():
            """Summary
            
            Returns
            -------
            TYPE
                Description
            """
            return (self._H_nonzero_row, self._H_nonzero_col)

        self._hessian_structure_lambda = hessian_structure

        msg = "Hessian of the Lagrangian lambda generated successfully."
        console_out(msg)

    # def shift_scale_variable_bounds(self):
    #   self._x_bnd_l = self._x_bnd_l - self.scaling.x_shifts
    #   self._x_bnd_u = self._x_bnd_u - self.scaling.x_shifts

    def check_nlp_functions(self):
        """Dumps values of the NLP callables evaluated at the initial guess.
        
        

        """
        if self.backend.ocp.settings.check_nlp_functions:
            print('\n\n\n')
            x_data = np.array(range(1, self.num_x + 1), dtype=float)
            # x_data = np.ones(self.num_x)
            
            print(f"x Variables:\n{self.backend.x_var}\n")
            print(f"x Data:\n{x_data}\n")

            # if self.optimal_control_problem.settings.derivative_level == 2:
            #     lagrange = np.array(range(1, self.num_c + 1), dtype=float)
            #     # lagrange = np.ones(self.num_c)
            #     obj_factor = 2.0
            #     # obj_factor = 1
            #     print(f"Objective Factor:\n{obj_factor}\n")
            #     print(f"Lagrange Multipliers:\n{lagrange}\n")

            J = self.backend.evaluate_J(x_data)
            print(f"J:\n{J}\n")

            g = self.backend.evaluate_g(x_data)
            print(f"g:\n{g}\n")

            c = self.backend.evaluate_c(x_data)
            print(f"c:\n{c}\n")

            G_struct = self.backend.evaluate_G_structure()
            print(f"G Structure:\n{G_struct[0]}\n\n{G_struct[1]}\n")

            G_nnz = self.backend.evaluate_G_num_nonzero()
            print(f"G Number Nonzeros:\n{G_nnz}\n")

            G_nz = self.backend.evaluate_G_nonzeros(x_data)
            print(f"G Nonzeros:\n{G_nz}\n")

            G = self.backend.evaluate_G(x_data)
            print(f"G:\n{G}\n")

            if self.optimal_control_problem.settings.derivative_level == 2:
                H_struct = self._hessian_structure_lambda()
                print(f"H Structure:\n{H_struct[0]}\n\n{H_struct[1]}\n")

                H_nnz = len(H_struct[0])
                print(f"H Nonzeros:\n{H_nnz}\n")

                H = self._hessian_lambda(x_data, lagrange, obj_factor)
                print(f"H:\n{H}\n")

            if self.optimal_control_problem.settings.dump_nlp_check_json:
                file_extension = ".json"
                filename_full = str(self.optimal_control_problem.settings.dump_nlp_check_json) + file_extension

                sG = sparse.coo_matrix((G, G_struct), shape=self._G_shape)
                
                data = {
                    "x": x_data.tolist(),
                    "J": float(J),
                    "g": np.array(g).tolist(),
                    "c": np.array(c).tolist(), 
                    "G_data": sG.data.tolist(),
                    "G_row": sG.row.tolist(),
                    "G_col": sG.col.tolist(),
                    "G_nnz": int(sG.nnz),
                    "num_x": int(self.num_x),
                    "num_c": int(self.num_c),
                    }

                if self.optimal_control_problem.settings.derivative_level == 2:
                    sH = sparse.coo_matrix((H, H_struct), shape=self._H_shape)
                    data["H_data"] = sH.data.flatten().tolist()
                    data["H_row"] = sH.row.flatten().tolist()
                    data["H_col"] = sH.col.flatten().tolist()
                    data["H_nnz"] = int(sH.nnz)

                    print(sH)

                with open(filename_full, "w", encoding="utf-8") as file:
                    json.dump(data, file, ensure_ascii=False, indent=4)

            print('\n\n\n')
            raise NotImplementedError
