from pathlib import Path
import re
from subprocess import run
import sys

from ..advisors import snippet_str_advisor
from .. import conf
from ..utils import layout_comment as layout, make_open_tmp_file

def _store_snippet(snippet):
    tmp_fh, fpath = make_open_tmp_file(conf.SNIPPET_FNAME, mode='w')
    tmp_fh.write(snippet)
    tmp_fh.close()
    return fpath

def _get_flake8_results(fpath):
    """
    Running flake8 from the terminal is much more straight forward than trying
    to run it through Python. The assumption here is that flake8's location is
    tightly coupled to the location of the python executable it is associated
    with. There may be better ways of finding flake8 (and the correct instance
    of flake8 for the Python version being run) so it can be invoked.
    """
    bin_path = Path(sys.executable).parent
    flake8_fpath = str(bin_path / 'flake8')
    cmd = [flake8_fpath, str(fpath)]
    res = run(args=cmd, capture_output=True)
    return res

@snippet_str_advisor(warning=True)
def lint_snippet(snippet):
    """
    Look for "lint" as defined by flake8 linter and share the results.
    """
    if not conf.INCLUDE_LINTING:  ## disabled when testing for speed reasons
        return None
    fpath = _store_snippet(snippet)
    res = _get_flake8_results(fpath)
    if not res.stdout:
        return None

    title = layout("""\

        ### Python code issues (found by flake8 linter)

        "Linters" are software tools. They detect everything from trivial style
        mistakes of no consequence to program behaviour through to show-stopper
        syntax errors.

        Software developers can be notoriously fussy about the smallest details
        of code styling and a linter can not only detect actual errors in code
        it can also prevent developers becoming completely distracted by trivial
        irritants. Distracted developers miss real issues with programs so it
        can be of practical importance to pick up the "small stuff". Plus it
        enables teams of programmers to work on the same code base without
        spending all their time restyling each other's code and arguing about
        "standards".

        Here is what the linter reported about your snippet:

        """)
    lint_lines = [line.strip()
        for line in str(res.stdout, encoding='utf-8').strip().split('\n')
        if line.strip()]
    ## /tmp/snippet.py:1:23: W291 trailing whitespace
    ## /tmp/snippet.py:4:1: E999 IndentationError: unexpected indent
    prog = re.compile(
        fr"""^{fpath}:
            (?P<{conf.LINT_LINE_NO}>\d+)       ## line_no = one or more digits after snippet.py: and before the next : 
            :\d+:\s{{1}}                       ## one or more digits, :, and one space
            (?P<{conf.LINT_MSG_TYPE}>\w{{1}})  ## msg_type = one letter
            \d{{1,3}}\s{{1}}                   ## 1-3 digits and one space
            (?P<{conf.LINT_MSG}>.*)            ## msg = everything else
        """, flags=re.VERBOSE)  # @UndefinedVariable
    lint_details = []
    for lint_line in lint_lines:
        result = prog.match(lint_line)
        lint_details.append(result.groupdict())
    lint_details.sort(
        key=lambda d: (d[conf.LINT_LINE_NO], d[conf.LINT_MSG_TYPE]))
    msg_lines = []
    for lint_detail in lint_details:
        line_no = lint_detail[conf.LINT_LINE_NO]
        if lint_detail[conf.LINT_MSG_TYPE] == 'E':
            msg_type = 'ERROR'
        elif lint_detail[conf.LINT_MSG_TYPE] == 'W':
            msg_type = 'Warning'
        else:
            msg_type = 'Other'
        msg = lint_detail[conf.LINT_MSG]
        msg_lines.append(f"Line {line_no:>3}: {msg_type} - {msg}")
    lint_message = layout('\n\n'.join(msg_lines))
    obviousness = layout("""\

        Linting is especially useful for an interpreted language like Python
        because there is no compiler to pick up "lint" errors. Linting is no
        substitute for unit testing though. And neither are a substitute for
        writing readable code that can be reasoned about with confidence - the
        single best protection against code not doing what it is meant to do.
        The goal should be code where there is obviously nothing wrong rather
        than code where there nothing obviously wrong.
        
        > "There are two ways of constructing a software design. One way is to
        make it so simple that there are obviously no deficiencies. And the
        other way is to make it so complicated that there are no obvious
        deficiencies." C.A.R. Hoare
        """)

    message = {
        conf.BRIEF: title + lint_message,
        conf.EXTRA: obviousness,
    }
    return message
