"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assertions_1 = require("aws-cdk-lib/assertions");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const lib_1 = require("../lib");
const defaults = require("@aws-solutions-constructs/core");
const test_helper_1 = require("./test-helper");
const ec2 = require("aws-cdk-lib/aws-ec2");
test('Test that properties are set correctly', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const construct = new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        }
    });
    expect(construct.kinesisFirehose).toBeDefined();
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.vpc).not.toBeDefined();
});
test('Test existing function is utilized correctly', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const testName = 'test-name';
    const existingFunction = new lambda.Function(stack, 'test-function', {
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        functionName: testName
    });
    new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        existingLambdaObj: existingFunction
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: testName
    });
    // This is 2 because there's a lambda function in the custom resource to
    // delete all the objects when cleaning up the s3 bucket in kinesisfirehose-s3
    template.resourceCountIs('AWS::Lambda::Function', 2);
});
test('Test that lambda function props are incorporated', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const testName = 'test-name';
    new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            functionName: testName
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: testName
    });
});
test('Test that new VPC is created', () => {
    // Stack
    const stack = new cdk.Stack();
    const cidrRange = '172.0.0.0/16';
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const construct = new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        deployVpc: true,
        vpcProps: {
            ipAddresses: ec2.IpAddresses.cidr(cidrRange)
        }
    });
    expect(construct.vpc).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: cidrRange
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".kinesis-firehose"
                ]
            ]
        },
    });
});
test('Test that existing VPC is used', () => {
    // Stack
    const stack = new cdk.Stack();
    const cidrInGetTestVpc = '172.168.0.0/16';
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const myVpc = defaults.getTestVpc(stack, false);
    const construct = new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingVpc: myVpc
    });
    expect(construct.vpc).toBeDefined();
    // Make sure we didn't deploy a new one anyway
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: cidrInGetTestVpc
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".kinesis-firehose"
                ]
            ]
        },
    });
});
test('Test fail if existingFirehose does not have a stream name', () => {
    // Stack
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const testName = 'test-name';
    const mutatedFirehose = defaults.overrideProps(destination.kinesisFirehose, {});
    mutatedFirehose.deliveryStreamName = undefined;
    const app = () => {
        new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: mutatedFirehose,
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_18_X,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                functionName: testName
            }
        });
    };
    expect(app).toThrowError(/existingKinesisFirehose must have a defined deliveryStreamName/);
});
test('Test fail Vpc check with vpcProps yet deployVpc is false', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const app = () => {
        new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: destination.kinesisFirehose,
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_18_X,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            deployVpc: false,
            vpcProps: {
                ipAddresses: ec2.IpAddresses.cidr("10.0.0.0/16"),
            },
        });
    };
    // Assertion
    expect(app).toThrowError('Error - If deployVpc is not true, then vpcProps is ignored');
});
test('Test fail Vpc check with vpcProps yet deployVpc is undefined', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const app = () => {
        new lib_1.LambdaToKinesisFirehose(stack, 'test-target', {
            existingKinesisFirehose: destination.kinesisFirehose,
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_18_X,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            vpcProps: {
                ipAddresses: ec2.IpAddresses.cidr("10.0.0.0/16"),
            },
        });
    };
    // Assertion
    expect(app).toThrowError('Error - If deployVpc is not true, then vpcProps is ignored');
});
test('Confirm CheckVpcProps() is being called', () => {
    const stack = new cdk.Stack();
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToKinesisFirehose(stack, 'test-lambda-kinesisfirehose', {
            existingKinesisFirehose: destination.kinesisFirehose,
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_18_X,
                handler: 'index.handler'
            },
            deployVpc: true,
            existingVpc: vpc
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const destination = test_helper_1.GetTestFirehoseDestination(stack, 'test-destination');
    const props = {
        existingKinesisFirehose: destination.kinesisFirehose,
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToKinesisFirehose(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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