# Sphinx extension to integrate IDF build system information
# into the Sphinx Build
#
# Runs early in the Sphinx process, runs CMake to generate the dummy IDF project
# in this directory - including resolving paths, etc.
#
# Then emits the new 'project-build-info' event which has information read from IDF
# build system, that other extensions can use to generate relevant data.
import json
import os.path
import shutil
import subprocess
import sys
from pathlib import Path

from sphinx.util import logging

# this directory also contains the dummy IDF project
project_path = os.path.abspath(os.path.dirname(__file__))

# Targets which needs --preview to build
PREVIEW_TARGETS = ['esp32p4', 'esp32c5']


class IdfBuilder():
    def __init__(self) -> None:
        self.project_description = {}
        self.logger = logging.getLogger(__name__)
        self.component_info_ignore_headers = []

    def add_ignored_headers(self, ignore_file):
        with open(ignore_file, 'r') as f:
            for line in f.readlines():
                if line.startswith('#'):  # Comment, do not add
                    continue
                self.component_info_ignore_headers.append(line.strip())

    def generate_idf_info(self, app, config):
        if 'doxygen_component_info' in config.idf_build_system and config.idf_build_system['doxygen_component_info']:
            config.run_doxygen_header_edit_callback = self.append_component_info
        if 'component_info_ignore_file' in config.idf_build_system and config.idf_build_system['component_info_ignore_file']:
            self.add_ignored_headers(config.idf_build_system['component_info_ignore_file'])

        # Set IDF_DOC_BUILD to trigger a doc build of ESP-IDF
        # Will force IDF to include components that might not be included by default
        os.environ['IDF_DOC_BUILD'] = 'y'

        print('Running CMake on dummy project to get build info...')

        if not app.config.idf_target:
            raise RuntimeError('A valid target is needed to build docs for ESP-IDF. '
                               'Please re-run build-docs with a target specified, e.g: '
                               'build-docs -t esp32')

        build_dir = os.path.dirname(app.doctreedir.rstrip(os.sep))
        cmake_build_dir = os.path.join(build_dir, 'build_dummy_project')
        idf_py_path = os.path.join(app.config.project_path, 'tools', 'idf.py')
        print('Running idf.py...')
        idf_py = [sys.executable,
                  idf_py_path,
                  '-B',
                  cmake_build_dir,
                  '-C',
                  project_path,
                  '-D',
                  'SDKCONFIG={}'.format(os.path.join(build_dir, 'dummy_project_sdkconfig'))
                  ]

        # force a clean idf.py build w/ new sdkconfig each time
        # (not much slower than 'reconfigure', avoids any potential config & build versioning problems
        shutil.rmtree(cmake_build_dir, ignore_errors=True)
        print('Starting new dummy IDF project... ')

        if (app.config.idf_target in PREVIEW_TARGETS):
            subprocess.check_call(idf_py + ['--preview', 'set-target', app.config.idf_target])
        else:
            subprocess.check_call(idf_py + ['set-target', app.config.idf_target])

        print('Running CMake on dummy project...')
        subprocess.check_call(idf_py + ['reconfigure'])

        with open(os.path.join(cmake_build_dir, 'project_description.json')) as f:
            self.project_description = json.load(f)
        if self.project_description['target'] != app.config.idf_target:
            # this shouldn't really happen unless someone has been moving around directories inside _build, as
            # the cmake_build_dir path should be target-specific
            raise RuntimeError(('Error configuring the dummy IDF project for {}. ' +
                                'Target in project description is {}. ' +
                                'Is build directory contents corrupt?')
                               .format(app.config.idf_target, self.project_description['target']))

        app.emit('project-build-info', self.project_description)

        return []

    def append_component_info(self, rst_output, header_file_path):
        """Appends build specific component info to the rst for the API-reference header include.

            Adds the following information by parsing the project_info from the dummy build:
                * Include path to the header file
                * REQUIRES information for the API

        Args:
            rst_output: rst_output already generated by run_doxygen
            header_file_path: path to header file which to add component specific info for

        Returns:
            rst output which will be used by run_doxygen to generate the .inc file for the header
        """

        def find_include_path(component_rel_header_file_path, component_rel_include_paths):
            """Get the shortest include path for the given header file.

            Args:
                component_rel_header_file_path: a path of the header file, relative to the component it belongs to
                component_rel_include_paths: list of component public include paths, relative to the component it self.

            Returns:
                shortest possible include path for the header
                None if no include path exists
            """
            matching_include_paths = [inc_path for inc_path in component_rel_include_paths if inc_path in component_rel_header_file_path.parents]

            if not matching_include_paths:
                return

            # If multiple include paths are possible we should show the shortest one, which well be
            # the one relative to the longest include dir path
            longest_include_dir_path = max(matching_include_paths, key=lambda path: len(str(path)))

            return component_rel_header_file_path.relative_to(longest_include_dir_path)

        def find_component(header_file_path):
            component_name = header_file_path.parts[1]

            if header_file_path.parts[0] != 'components':
                self.logger.warning(f'Failed to find component name for header path: {header_file_path}')

            return component_name

        if header_file_path in self.component_info_ignore_headers:
            return rst_output

        header_file_path = Path(header_file_path)

        component_name = find_component(header_file_path)

        if component_name not in self.project_description['build_component_info']:
            self.logger.warning(f'Component {component_name} from header file {header_file_path} '
                                f'not found in project_description[build_component_info]')
            return rst_output

        component_info = self.project_description['build_component_info'][component_name]
        component_include_dirs = component_info['include_dirs']
        if component_include_dirs is None:
            self.logger.warning(f'Component {component_name} has no public include directories')

        # convert all include paths to be relative to the component
        component_include_dirs = [Path(inc_path) for inc_path in component_include_dirs]
        component_rel_include_paths = [inc_path.relative_to(component_info['dir']) if inc_path.is_absolute() else inc_path
                                       for inc_path in component_include_dirs]

        rel_include_path = find_include_path(header_file_path.relative_to(Path('components') / component_name), component_rel_include_paths)

        if not rel_include_path:
            self.logger.warning(f'Could not find a include path for {header_file_path} in component {component_name}')

        rst_output += (f'* This header file can be included with:\n'
                       f'\n'
                       f'   .. code-block:: c\n'
                       f'\n'
                       f'       #include "{rel_include_path}"\n'
                       f'\n')

        if component_name not in self.project_description['common_component_reqs']:
            rst_output += (f'* This header file is a part of the API provided by the ``{component_name}`` component.'
                           f' To declare that your component depends on ``{component_name}``, add the following to your CMakeLists.txt:\n'
                           f'\n'
                           f'   .. code-block:: none\n'
                           f'\n'
                           f'       REQUIRES {component_name}\n'
                           f'\n'
                           f'   or\n'
                           f'\n'
                           f'   .. code-block:: none\n'
                           f'\n'
                           f'       PRIV_REQUIRES {component_name}\n'
                           f'\n'
                           )

        rst_output += '\n'

        return rst_output


idf_builder = IdfBuilder()


def setup(app):
    # Setup some common paths

    try:
        build_dir = os.environ['BUILDDIR']  # TODO see if we can remove this
    except KeyError:
        build_dir = os.path.dirname(app.doctreedir.rstrip(os.sep))

    try:
        os.mkdir(build_dir)
    except OSError:
        pass

    try:
        os.mkdir(os.path.join(build_dir, 'inc'))
    except OSError:
        pass

    # Fill in a default IDF_PATH if it's missing (ie when Read The Docs is building the docs)
    try:
        idf_path = os.environ['IDF_PATH']
    except KeyError:
        idf_path = os.path.realpath(os.path.join(os.path.dirname(__file__), '..'))

    app.add_config_value('docs_root', os.path.join(idf_path, 'docs'), 'env')
    app.add_config_value('idf_path', idf_path, 'env')
    app.add_config_value('idf_build_system', {'doxygen_component_info': False,
                                              'component_info_ignore_file': None}, 'env')

    app.add_event('project-build-info')

    # we want this to run early in the docs build but unclear exactly when
    app.connect('config-inited', idf_builder.generate_idf_info)

    return {'parallel_read_safe': True, 'parallel_write_safe': True, 'version': '0.1'}
