"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reset = exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_sdk_1 = require("aws-sdk");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const clients = new Map();
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return withFakeHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        const language = env_lambda_shared_1.requireEnv('TARGET_LANGUAGE');
        const created = new Array();
        for (const snsRecord of event.Records) {
            const s3Event = JSON.parse(snsRecord.Sns.Message);
            for (const record of s3Event.Records) {
                // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
                // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
                // that would not replace `@` in the position where it is in the keys... So we have to work on
                // the URI components instead.
                const inputKey = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
                const [, packageName, packageVersion] = (_a = inputKey.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
                if (packageName == null) {
                    throw new Error(`Invalid object key: "${inputKey}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
                }
                const packageFqn = `${packageName}@${packageVersion}`;
                const client = (clients.has(record.awsRegion)
                    ? clients
                    : clients.set(record.awsRegion, new aws_sdk_1.S3({ region: record.awsRegion }))).get(record.awsRegion);
                console.log(`Source Bucket:  ${record.s3.bucket.name}`);
                console.log(`Source Key:     ${inputKey}`);
                console.log(`Source Version: ${record.s3.object.versionId}`);
                console.log(`Fetching assembly: ${inputKey}`);
                const assemblyResponse = await client.getObject({ Bucket: record.s3.bucket.name, Key: inputKey }).promise();
                if (!assemblyResponse.Body) {
                    throw new Error(`Response body for assembly at key ${inputKey} is empty`);
                }
                const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
                const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
                if (language !== 'typescript' && assembly.targets[language] == null) {
                    console.error(`Package ${assembly.name}@${assembly.version} does not support ${language}, skipping!`);
                    console.log(`Assembly targets: ${JSON.stringify(assembly.targets, null, 2)}`);
                    for (const submodule of [undefined, ...submodules]) {
                        const key = inputKey.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(language), submodule)) + constants.NOT_SUPPORTED_SUFFIX;
                        const response = await uploadFile(client, context, record.s3.bucket.name, key, record.s3.object.versionId);
                        created.push({ bucket: record.s3.bucket.name, key, versionId: response.VersionId });
                    }
                    continue;
                }
                await generateDocs(language);
                async function generateDocs(lang) {
                    const uploads = new Map();
                    const docs = await docgen.Documentation.forPackage(packageFqn, { language: docgen.Language.fromString(lang) });
                    function renderAndDispatch(submodule) {
                        console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                        const page = docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render();
                        const key = inputKey.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(lang), submodule));
                        console.log(`Uploading ${key}`);
                        const upload = uploadFile(client, context, record.s3.bucket.name, key, record.s3.object.versionId, page);
                        uploads.set(key, upload);
                    }
                    renderAndDispatch();
                    for (const submodule of submodules) {
                        renderAndDispatch(submodule);
                    }
                    for (const [key, upload] of uploads.entries()) {
                        const response = await upload;
                        created.push({ bucket: record.s3.bucket.name, key: key, versionId: response.VersionId });
                        console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
                    }
                }
            }
        }
        return created;
    });
}
exports.handler = handler;
async function withFakeHome(cb) {
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
    }
}
function uploadFile(s3, context, bucket, key, sourceVersionId, body) {
    return s3.putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: 'public',
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
        },
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
/**
 * Visible for testing. Clears the caches so that the next execution runs clean.
 */
function reset() {
    clients.clear();
}
exports.reset = reset;
//# sourceMappingURL=data:application/json;base64,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