"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const s3n = require("@aws-cdk/aws-s3-notifications");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("./transliterator");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT];
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.alarmsDeadLetterQueueNotEmpty = new Map();
        const repository = props.vpcEndpoints
            ? (_a = props.codeArtifact) === null || _a === void 0 ? void 0 : _a.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        const topic = new sns.Topic(this, 'Topic');
        // multiplex the event via an sns topic so that all functions get it
        bucket.addEventNotification(aws_s3_1.EventType.OBJECT_CREATED, new s3n.SnsDestination(topic), {
            prefix: constants.STORAGE_KEY_PREFIX,
            suffix: constants.ASSEMBLY_KEY_SUFFIX,
        });
        for (const lang of SUPPORTED_LANGUAGES) {
            const lambda = new transliterator_1.Transliterator(this, `${lang.toString()}Handler`, {
                deadLetterQueueEnabled: true,
                description: `Creates ${lang} documentation from jsii-enabled npm packages`,
                environment: { ...environment, TARGET_LANGUAGE: lang.toString() },
                logRetention: (_b = props.logRetention) !== null && _b !== void 0 ? _b : aws_logs_1.RetentionDays.TEN_YEARS,
                memorySize: 10240,
                retryAttempts: 2,
                timeout: core_1.Duration.minutes(15),
                vpc: props.vpc,
                vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : { subnetType: aws_ec2_1.SubnetType.ISOLATED },
            });
            repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(lambda);
            // The handler reads & writes to this bucket.
            bucket.grantRead(lambda, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
            bucket.grantWrite(lambda, `${constants.STORAGE_KEY_PREFIX}*${constants.DOCS_KEY_SUFFIX_ANY}`);
            // subscribe to the topic
            lambda.addEventSource(new aws_lambda_event_sources_1.SnsEventSource(topic, { deadLetterQueue: lambda.deadLetterQueue }));
            props.monitoring.watchful.watchLambdaFunction('Transliterator Function', lambda);
            this.alarmsDeadLetterQueueNotEmpty.set(lang, lambda.deadLetterQueue.metricApproximateNumberOfMessagesVisible()
                .createAlarm(this, `${lang}DLQAlarm`, {
                alarmDescription: `The ${lang} transliteration function failed for one or more packages`,
                comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold: 1,
            }));
        }
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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