"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const path_1 = require("path");
const url_1 = require("url");
const zlib_1 = require("zlib");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws.s3().getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        }).promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        const tar = await gunzip(Buffer.from(tarball.Body));
        const { dotJsii, licenseText, packageJson } = await new Promise((ok, ko) => {
            let dotJsiiBuffer;
            let licenseTextBuffer;
            let packageJsonData;
            const extractor = tar_stream_1.extract({ filenameEncoding: 'utf-8' })
                .once('error', (reason) => {
                ko(reason);
            })
                .once('finish', () => {
                if (dotJsiiBuffer == null) {
                    ko(new Error('No .jsii file found in tarball!'));
                }
                else if (packageJsonData == null) {
                    ko(new Error('No package.json file found in tarball!'));
                }
                else {
                    ok({ dotJsii: dotJsiiBuffer, licenseText: licenseTextBuffer, packageJson: packageJsonData });
                }
            })
                .on('entry', (headers, stream, next) => {
                const chunks = new Array();
                if (headers.name === 'package/.jsii') {
                    return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('error', ko)
                        .once('end', () => {
                        dotJsiiBuffer = Buffer.concat(chunks);
                        // Skip on next runLoop iteration so we avoid filling the stack.
                        setImmediate(next);
                    })
                        .resume();
                }
                else if (headers.name === 'package/package.json') {
                    return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('error', ko)
                        .once('end', () => {
                        packageJsonData = Buffer.concat(chunks);
                        // Skip on next runLoop iteration so we avoid filling the stack.
                        setImmediate(next);
                    })
                        .resume();
                }
                else if (isLicenseFile(headers.name)) {
                    return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('error', ko)
                        .once('end', () => {
                        licenseTextBuffer = Buffer.concat(chunks);
                        // Skip on next runLoop iteration so we avoid filling the stack.
                        setImmediate(next);
                    })
                        .resume();
                }
                // Skip on next runLoop iteration so we avoid filling the stack.
                return setImmediate(next);
            });
            extractor.write(tar, (err) => {
                if (err != null) {
                    ko(err);
                }
                extractor.end();
            });
        });
        const metadata = { date: payload.time, licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8') };
        const { license: packageLicense, name: packageName, version: packageVersion } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense } = JSON.parse(packageJson.toString('utf-8'));
        if (packageJsonName !== packageName || packageJsonVersion !== packageVersion || packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const assemblyKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`;
        console.log(`Writing assembly at ${assemblyKey}`);
        const packageKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.PACKAGE_KEY_SUFFIX}`;
        console.log(`Writing package at  ${packageKey}`);
        const metadataKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.METADATA_KEY_SUFFIX}`;
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws.s3().putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        }).promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        result.push(created);
    }
    return result;
});
function gunzip(data) {
    const chunks = new Array();
    return new Promise((ok, ko) => zlib_1.createGunzip()
        .once('error', ko)
        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
        .once('end', () => ok(Buffer.concat(chunks)))
        .end(data));
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return possibleExtensions.has(ext.toLowerCase())
        && path_1.basename(fileName, ext).toUpperCase() === 'LICENSE';
}
//# sourceMappingURL=data:application/json;base64,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