"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const METRICS_NAMESPACE = 'ConstructHub/CatalogBuilder';
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    console.log('Loading existing catalog...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    if (!data.Body) {
        console.log('Catalog not found. Recreating...');
        const failures = {};
        for await (const object of relevantObjects(BUCKET_NAME)) {
            const docsKey = object.Key;
            try {
                await appendPackage(packages, docsKey, BUCKET_NAME);
            }
            catch (e) {
                failures[docsKey] = e;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setNamespace(METRICS_NAMESPACE);
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric('FailedPackagesOnRecreation', failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    else {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
        console.log('Registering new packages...');
        for (const record of event.Records) {
            // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
            // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
            // that would not replace `@` in the position where it is in the keys... So we have to work on
            // the URI components instead.
            const key = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
            // note that we intentionally don't catch errors here to let these
            // event go to the DLQ for manual inspection.
            await appendPackage(packages, key, BUCKET_NAME);
        }
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setNamespace(METRICS_NAMESPACE);
        metrics.putMetric('RegisteredPackagesMajorVersion', catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Upload the result to S3 and exit.
    return aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: 'public, max-age=300',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: constants.STORAGE_KEY_PREFIX };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.DOCS_KEY_SUFFIX_TYPESCRIPT))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, docsKey, bucketName) {
    var _a, _b, _c;
    console.log(`Processing key: ${docsKey}`);
    const pkgKey = docsKey.replace(constants.DOCS_KEY_SUFFIX_TYPESCRIPT, constants.PACKAGE_KEY_SUFFIX);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    if (found != null && version.compare(found.version) <= 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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