"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const stream_1 = require("stream");
const zip = require("zlib");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const tar = require("tar-stream");
const catalog_builder_lambda_1 = require("../../../backend/catalog-builder/catalog-builder.lambda");
const aws = require("../../../backend/shared/aws.lambda-shared");
const constants = require("../../../backend/shared/constants");
let mockBucketName;
beforeEach((done) => {
    process.env.BUCKET_NAME = mockBucketName = crypto_1.randomBytes(16).toString('base64');
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    aws.reset();
    process.env.BUCKET_NAME = mockBucketName = undefined;
    done();
});
test('no indexed packages', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
        }
        catch (e) {
            return cb(e);
        }
        return cb(new NoSuchKeyError());
    });
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
            expect(req.ContinuationToken).toBeUndefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, {});
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('application/json');
            expect(req.CacheControl).toBe('public, max-age=300');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({ Records: [] }, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('initial build', () => {
    // GIVEN
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            return cb(null, { Body: JSON.stringify(npmMetadata) });
        }
        const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    // this is the suffix that triggers the catalog builder.
    const docsSuffix = constants.DOCS_KEY_SUFFIX_TYPESCRIPT;
    const mockFirstPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${docsSuffix}` },
    ];
    const mockSecondPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${docsSuffix}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${docsSuffix}` },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('application/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '3');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '2.0.0-pre',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({ Records: [] }, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('incremental build', () => {
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            return cb(null, { Body: JSON.stringify(npmMetadata) });
        }
        const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else if (req.Key === constants.CATALOG_KEY) {
            return cb(null, {
                Body: JSON.stringify({
                    packages: [
                        {
                            description: 'Package @scope/package, version 2.3.4',
                            languages: { foo: 'bar' },
                            major: 2,
                            metadata: npmMetadata,
                            name: '@scope/package',
                            version: '2.3.4',
                        },
                        {
                            description: 'Package name, version 1.0.0',
                            languages: { foo: 'bar' },
                            major: 1,
                            metadata: npmMetadata,
                            name: 'name',
                            version: '1.0.0',
                        },
                        {
                            description: 'Package name, version 2.0.0-pre.10',
                            languages: { foo: 'bar' },
                            major: 2,
                            metadata: npmMetadata,
                            name: 'name',
                            version: '2.0.0-pre.10',
                        },
                    ],
                    updatedAt: new Date().toISOString(),
                }, null, 2),
            });
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    function createRecord(key) {
        return { s3: { object: { key } } };
    }
    const event = {
        Records: [
            createRecord(`${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}`),
            createRecord(`${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}`),
            createRecord(`${constants.STORAGE_KEY_PREFIX}@scope/package/v2.0.5${constants.PACKAGE_KEY_SUFFIX}`),
            createRecord(`${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre.1${constants.PACKAGE_KEY_SUFFIX}`),
        ],
    };
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('application/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '4');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 2.3.4',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '2.3.4',
                },
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre.10',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '2.0.0-pre.10',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler(event, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
class NoSuchKeyError extends Error {
    constructor() {
        super(...arguments);
        this.code = 'NoSuchKey';
        this.time = new Date();
    }
}
function mockNpmPackage(name, version) {
    const packageJson = {
        name,
        version,
        description: `Package ${name}, version ${version}`,
        jsii: {
            targets: { foo: 'bar' },
        },
    };
    const tarball = tar.pack();
    tarball.entry({ name: 'package/ignore-me.txt' }, 'Ignore Me!');
    tarball.entry({ name: 'package/package.json' }, JSON.stringify(packageJson, null, 2));
    tarball.finalize();
    const gzip = zip.createGzip();
    tarball.pipe(gzip);
    const passthrough = new stream_1.PassThrough();
    gzip.pipe(passthrough);
    return new Promise((ok) => {
        const chunks = new Array();
        passthrough.on('data', (chunk) => chunks.push(Buffer.from(chunk)));
        passthrough.once('end', () => {
            ok(Buffer.concat(chunks));
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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