"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/*
 * Creates an Appsync GraphQL API and with multiple tables.
 * Testing for importing, querying, and mutability.
 *
 * Stack verification steps:
 * Add to a table through appsync GraphQL API.
 * Read from a table through appsync API.
 *
 * -- aws appsync list-graphql-apis                 -- obtain apiId               --
 * -- aws appsync get-graphql-api --api-id [apiId]  -- obtain GraphQL endpoint    --
 * -- aws appsync list-api-keys --api-id [apiId]    -- obtain api key             --
 * -- bash verify.integ.graphql.sh [apiKey] [url]   -- shows query and mutation   --
 */
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphQLApi(stack, 'Api', {
    name: 'demoapi',
    schema: lib_1.Schema.fromAsset(path_1.join(__dirname, 'integ.graphql.graphql')),
    authorizationConfig: {
        defaultAuthorization: {
            authorizationType: lib_1.AuthorizationType.USER_POOL,
            userPoolConfig: {
                userPool,
                defaultAction: lib_1.UserPoolDefaultAction.ALLOW,
            },
        },
        additionalAuthorizationModes: [
            {
                authorizationType: lib_1.AuthorizationType.API_KEY,
            },
        ],
    },
});
const noneDS = api.addNoneDataSource('none', { name: 'None' });
noneDS.createResolver({
    typeName: 'Query',
    fieldName: 'getServiceVersion',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: '2017-02-28',
    })),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: 'v1',
    })),
});
const customerTable = new aws_dynamodb_1.Table(stack, 'CustomerTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const orderTable = new aws_dynamodb_1.Table(stack, 'OrderTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'customer',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    sortKey: {
        name: 'order',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
new aws_dynamodb_1.Table(stack, 'PaymentTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const paymentTable = aws_dynamodb_1.Table.fromTableName(stack, 'ImportedPaymentTable', 'PaymentTable');
const customerDS = api.addDynamoDbDataSource('customerDs', customerTable, { name: 'Customer' });
const orderDS = api.addDynamoDbDataSource('orderDs', orderTable, { name: 'Order' });
const paymentDS = api.addDynamoDbDataSource('paymentDs', paymentTable, { name: 'Payment' });
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomers',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').is('id'), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomerWithFirstOrder',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey
        .partition('order').auto()
        .sort('customer').is('customer.id'), lib_1.Values
        .projecting('order')
        .attribute('referral').is('referral')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'removeCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbDeleteItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const ops = [
    { suffix: 'Eq', op: lib_1.KeyCondition.eq },
    { suffix: 'Lt', op: lib_1.KeyCondition.lt },
    { suffix: 'Le', op: lib_1.KeyCondition.le },
    { suffix: 'Gt', op: lib_1.KeyCondition.gt },
    { suffix: 'Ge', op: lib_1.KeyCondition.ge },
];
for (const { suffix, op } of ops) {
    orderDS.createResolver({
        typeName: 'Query',
        fieldName: 'getCustomerOrders' + suffix,
        requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(op('customer', 'customer')),
        responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
    });
}
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersFilter',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.beginsWith('order', 'order'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersBetween',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.between('order', 'order1', 'order2'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
paymentDS.createResolver({
    typeName: 'Query',
    fieldName: 'getPayment',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
paymentDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'savePayment',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('payment')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const httpDS = api.addHttpDataSource('ds', 'https://aws.amazon.com/', { name: 'http' });
httpDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'doPostOnAws',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(`{
    "version": "2018-05-29",
    "method": "POST",
    # if full path is https://api.xxxxxxxxx.com/posts then resourcePath would be /posts
    "resourcePath": "/path/123",
    "params":{
        "body": $util.toJson($ctx.args),
        "headers":{
            "Content-Type": "application/json",
            "Authorization": "$ctx.request.headers.Authorization"
        }
    }
  }`),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(`
    ## Raise a GraphQL field error in case of a datasource invocation error
    #if($ctx.error)
      $util.error($ctx.error.message, $ctx.error.type)
    #end
    ## if the response status code is not 200, then return an error. Else return the body **
    #if($ctx.result.statusCode == 200)
        ## If response is 200, return the body.
        $ctx.result.body
    #else
        ## If response is not 200, append the response to error block.
        $utils.appendError($ctx.result.body, "$ctx.result.statusCode")
    #end
  `),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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