"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const db = require("@aws-cdk/aws-dynamodb");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const ObjectType = require("./object-type-definitions");
const ScalarType = require("./scalar-type-defintions");
/*
 * Creates an Appsync GraphQL API and schema in a code-first approach.
 *
 * Stack verification steps:
 * Deploy stack, get api key and endpoinScalarType. Check if schema connects to data source.
 *
 * -- bash verify.integ.graphql-schema.sh --start                 -- start                    --
 * -- aws appsync list-graphql-apis                               -- obtain apiId & endpoint  --
 * -- aws appsync list-api-keys --api-id [apiId]                  -- obtain api key           --
 * -- bash verify.integ.graphql-schema.sh --check [apiKey] [url]  -- check if success         --
 * -- bash verify.integ.graphql-schema.sh --clean                 -- clean                    --
 */
const app = new cdk.App();
const stack = new cdk.Stack(app, 'code-first-schema');
const schema = new appsync.Schema();
const node = schema.addType(new appsync.InterfaceType('Node', {
    definition: {
        created: ScalarType.string,
        edited: ScalarType.string,
        id: ScalarType.required_id,
    },
}));
const api = new appsync.GraphQLApi(stack, 'code-first-api', {
    name: 'api',
    schema: schema,
});
const planet = ObjectType.planet;
schema.addToSchema(planet.toString());
api.addType(new appsync.ObjectType('Species', {
    interfaceTypes: [node],
    definition: {
        name: ScalarType.string,
        classification: ScalarType.string,
        designation: ScalarType.string,
        averageHeight: ScalarType.float,
        averageLifespan: ScalarType.int,
        eyeColors: ScalarType.list_string,
        hairColors: ScalarType.list_string,
        skinColors: ScalarType.list_string,
        language: ScalarType.string,
        homeworld: planet.attribute(),
    },
}));
api.addToSchema('type Query {\n  getPlanets: [Planet]\n}', '\n');
const table = new db.Table(stack, 'table', {
    partitionKey: {
        name: 'id',
        type: db.AttributeType.STRING,
    },
});
const tableDS = api.addDynamoDbDataSource('planets', table);
tableDS.createResolver({
    typeName: 'Query',
    fieldName: 'getPlanets',
    requestMappingTemplate: appsync.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: appsync.MappingTemplate.dynamoDbResultList(),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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