"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/*
 * Creates an Appsync GraphQL API and Lambda with IAM Roles.
 * Testing for IAM Auth and grantFullAccess.
 *
 * Stack verification steps:
 * Install dependencies and deploy integration test. Invoke Lambda
 * function with different permissions to test policies.
 *
 * -- bash verify.integ.graphql-iam.sh --start             -- get dependencies/deploy    --
 * -- aws lambda list-functions                            -- obtain testFail/testQuery  --
 * -- aws lambda invoke /dev/stdout --function-name [FAIL] -- fails beacuse no IAM Role` --
 * -- aws lambda invoke /dev/stdout --function-name [Query]-- succeeds with empty get  ` --
 * -- bash verify.integ.graphql-iam.sh --clean             -- clean dependencies/deploy  --
 */
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphQLApi(stack, 'Api', {
    name: 'Integ_Test_IAM',
    schema: lib_1.Schema.fromAsset(path_1.join(__dirname, 'integ.graphql-iam.graphql')),
    authorizationConfig: {
        defaultAuthorization: {
            authorizationType: lib_1.AuthorizationType.USER_POOL,
            userPoolConfig: {
                userPool,
                defaultAction: lib_1.UserPoolDefaultAction.ALLOW,
            },
        },
        additionalAuthorizationModes: [
            {
                authorizationType: lib_1.AuthorizationType.IAM,
            },
        ],
    },
});
const testTable = new aws_dynamodb_1.Table(stack, 'TestTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const testDS = api.addDynamoDbDataSource('ds', testTable, { name: 'testDataSource' });
testDS.createResolver({
    typeName: 'Query',
    fieldName: 'getTest',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
testDS.createResolver({
    typeName: 'Query',
    fieldName: 'getTests',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
testDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addTest',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('test')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const lambdaIAM = new aws_iam_1.Role(stack, 'LambdaIAM', { assumedBy: new aws_iam_1.ServicePrincipal('lambda') });
api.grant(lambdaIAM, lib_1.IamResource.custom('types/Query/fields/getTests'), 'appsync:graphql');
api.grant(lambdaIAM, lib_1.IamResource.ofType('test'), 'appsync:GraphQL');
api.grantMutation(lambdaIAM, 'addTest');
new aws_lambda_1.Function(stack, 'testQuery', {
    code: aws_lambda_1.Code.fromAsset('verify'),
    handler: 'iam-query.handler',
    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
    environment: { APPSYNC_ENDPOINT: api.graphQlUrl },
    role: lambdaIAM,
});
new aws_lambda_1.Function(stack, 'testFail', {
    code: aws_lambda_1.Code.fromAsset('verify'),
    handler: 'iam-query.handler',
    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
    environment: { APPSYNC_ENDPOINT: api.graphQlUrl },
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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