"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const t = require("./scalar-type-defintions");
// Schema Definitions
const type = 'type test {\n  version: String!\n}\n\n';
const query = 'type Query {\n  getTests: [ test! ]!\n}\n\n';
const mutation = 'type Mutation {\n  addTest(version: String!): test\n}\n';
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
describe('basic testing schema definition mode `code`', () => {
    test('definition mode `code` produces empty schema definition', () => {
        // WHEN
        new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: '',
        });
    });
    test('definition mode `code` generates correct schema with addToSchema', () => {
        // WHEN
        const api = new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
        });
        api.addToSchema(type);
        api.addToSchema(query);
        api.addToSchema(mutation);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${type}\n${query}\n${mutation}\n`,
        });
    });
});
describe('testing schema definition mode `file`', () => {
    test('definition mode `file` produces correct output', () => {
        // WHEN
        new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${type}${query}${mutation}`,
        });
    });
    test('definition mode `file` errors when addObjectType is called', () => {
        // WHEN
        const api = new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addType(new appsync.ObjectType('blah', {
                definition: { fail: t.id },
            }));
        }).toThrowError('API cannot add type because schema definition mode is not configured as CODE.');
    });
    test('definition mode `file` errors when addInterfaceType is called', () => {
        // WHEN
        const api = new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addType(new appsync.InterfaceType('blah', {
                definition: { fail: t.id },
            }));
        }).toThrowError('API cannot add type because schema definition mode is not configured as CODE.');
    });
    test('definition mode `file` errors when addToSchema is called', () => {
        // WHEN
        const api = new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addToSchema('blah');
        }).toThrowError('API cannot append to schema because schema definition mode is not configured as CODE.');
    });
});
//# sourceMappingURL=data:application/json;base64,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