"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const t = require("./scalar-type-defintions");
let stack;
let api;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    api = new appsync.GraphQLApi(stack, 'api', {
        name: 'api',
    });
});
describe('testing Object Type properties', () => {
    test('ObjectType can implement from interface types', () => {
        // WHEN
        const baseTest = new appsync.InterfaceType('baseTest', {
            definition: {
                id: t.id,
            },
        });
        const objectTest = new appsync.ObjectType('objectTest', {
            interfaceTypes: [baseTest],
            definition: {
                id2: t.id,
            },
            directives: [appsync.Directive.custom('@test')],
        });
        api.addToSchema(baseTest.toString());
        api.addToSchema(objectTest.toString());
        const gql_interface = 'interface baseTest {\n  id: ID\n}\n';
        const gql_object = 'type objectTest implements baseTest @test {\n  id2: ID\n  id: ID\n}\n';
        const out = `${gql_interface}${gql_object}`;
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('ObjectType can implement from multiple interface types', () => {
        // WHEN
        const baseTest = new appsync.InterfaceType('baseTest', {
            definition: { id: t.id },
        });
        const anotherTest = new appsync.InterfaceType('anotherTest', {
            definition: { id2: t.id },
        });
        const objectTest = new appsync.ObjectType('objectTest', {
            interfaceTypes: [anotherTest, baseTest],
            definition: {
                id3: t.id,
            },
        });
        api.addToSchema(baseTest.toString());
        api.addToSchema(anotherTest.toString());
        api.addToSchema(objectTest.toString());
        const gql_interface = 'interface baseTest {\n  id: ID\n}\ninterface anotherTest {\n  id2: ID\n}\n';
        const gql_object = 'type objectTest implements anotherTest, baseTest {\n  id3: ID\n  id2: ID\n  id: ID\n}\n';
        const out = `${gql_interface}${gql_object}`;
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can be a Graphql Type', () => {
        // WHEN
        const baseTest = new appsync.ObjectType('baseTest', {
            definition: {
                id: t.id,
            },
        });
        const graphqlType = baseTest.attribute();
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: graphqlType,
            },
        });
        api.addToSchema(test.toString());
        const out = 'type Test {\n  test: baseTest\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can implement Resolvable Field in definition', () => {
        // WHEN
        const field = new appsync.ResolvableField({
            returnType: t.string,
            dataSource: api.addNoneDataSource('none'),
            args: {
                arg: t.int,
            },
        });
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
                resolve: field,
            },
        });
        api.addToSchema(test.toString());
        const out = 'type Test {\n  test: String\n  resolve(arg: Int): String\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can implement Resolvable Field from GraphqlType', () => {
        // WHEN
        const field = new appsync.ResolvableField({
            returnType: t.string,
            dataSource: api.addNoneDataSource('none'),
            args: {
                arg: t.int,
            },
        });
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
                resolve: field,
            },
        });
        api.addToSchema(test.toString());
        const out = 'type Test {\n  test: String\n  resolve(arg: Int): String\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('Object Type can implement Resolvable Field for pipelineResolvers', () => {
        // WHEN
        const test = new appsync.ObjectType('Test', {
            definition: {
                resolve: new appsync.ResolvableField({
                    returnType: t.string,
                    dataSource: api.addNoneDataSource('none'),
                    args: {
                        arg: t.int,
                    },
                    pipelineConfig: ['test', 'test'],
                }),
            },
        });
        api.addToSchema(test.toString());
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::Resolver', {
            Kind: 'PIPELINE',
            PipelineConfig: { Functions: ['test', 'test'] },
        });
    });
    test('Object Type can dynamically add Fields', () => {
        // WHEN
        const field = new appsync.ResolvableField({
            returnType: t.string,
            dataSource: api.addNoneDataSource('none'),
            args: {
                arg: t.int,
            },
        });
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
            },
        });
        test.addField('resolve', field);
        // test.addField('resolve', field);
        test.addField('dynamic', t.string);
        api.addToSchema(test.toString());
        const out = 'type Test {\n  test: String\n  resolve(arg: Int): String\n  dynamic: String\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
        expect(stack).toHaveResource('AWS::AppSync::Resolver');
    });
    test('Object Type can dynamically add Fields', () => {
        // WHEN
        const garbage = new appsync.InterfaceType('Garbage', {
            definition: {
                garbage: t.string,
            },
        });
        const test = new appsync.ObjectType('Test', {
            definition: {
                test: t.string,
            },
        });
        const field = new appsync.ResolvableField({
            returnType: garbage.attribute(),
            dataSource: api.addNoneDataSource('none'),
            args: {
                arg: garbage.attribute(),
            },
        });
        test.addField('resolve', field);
        // test.addField('resolve', field);
        test.addField('dynamic', garbage.attribute());
        api.addToSchema(test.toString());
        const out = 'type Test {\n  test: String\n  resolve(arg: Garbage): Garbage\n  dynamic: Garbage\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
        expect(stack).toHaveResource('AWS::AppSync::Resolver');
    });
});
//# sourceMappingURL=data:application/json;base64,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